(ns burningswell.web.modules.settings
  "The settings page."
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [clojure.core.async :refer [<! #?(:clj go)]]
            [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.dom :as dom]
            [burningswell.web.getter.settings :as settings]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.system.core :as system]
            [burningswell.web.ui.layout :refer [layout]]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(def ^:dynamic *units*
  [{:id :eu
    :description "Europe"
    :units ["meters" "kilometers per hour" "Celsius"]}
   {:id :uk
    :description "United Kingdom"
    :units ["feet" "miles per hour" "Celsius"]}
   {:id :us
    :description "United States"
    :units ["feet" "miles per hour" "Fahrenheit"]}])

(defn change-value [system key]
  (fn [event]
    ((system/change-value system key) event)
    (let [{:keys [user settings]} (system/evaluate system settings/page)]
      (go (let [{:keys [body status]} (<! (api/save-user-settings system user settings))]
            (when (= status 200)
              (system/dispatch! system :user/settings body)))))))

(rum/defc geo-location < rum/static
  "Render the content of the settings page."
  [system page]
  [:div.settings__geo-location
   [:h6.settings__geo-location-headline
    "Use Geo Location of device"]
   [:div.settings__geo-location-select
    (mdl/switch
     {:checked false
      :for "geo-location"
      :input {:id "geo-location"}
      :mdl [:ripple]
      :on-change (change-value system :settings/geo-enabled)})]])

(rum/defc unit-headline
  "Render the headline of a measurement unit."
  [content]
  [:h6.settings__unit-label-headline content])

(rum/defc unit-description
  "Render the description of a measurement unit."
  [[waves wind temperature]]
  [:p.settings__unit-label-description
   "Show wave size in " [:i waves]
   ", wind speed in " [:i wind]
   " and temperature in " [:i temperature] "."])

(rum/defc unit-label
  "Render the label of a measurement unit."
  [description units]
  [:div.settings__unit-label
   (unit-headline description)
   (unit-description units)])

(defn- unit-selection
  "Render the selection of a measurement unit."
  [system key description units & [checked]]
  (let [id (str "unit-selection-" (name key))]
    (mdl/radio
     {:class "settings__unit-selection"
      ;; TODO: Wait for upstream release.
      ;; :checked checked
      :for id
      :input {:id id :name "units" :value (name key)}
      :label (unit-label description units)
      :mdl [:ripple]
      :on-change (change-value system :settings/units)})))

(defn units-checked? [user units]
  (= (or (-> user :_embedded :settings :units) "eu") (name units)))

(rum/defc units < rum/static
  "Render the content of the settings page."
  [system {:keys [user] :as page}]
  [:div.settings__units
   [:h6.settings__units-headline
    "Forecast Units"]
   [:div.settings__units-select
    (for [{:keys [id description units]} *units*]
      (unit-selection system id description units
                      (units-checked? user id)))]])

(rum/defc headline []
  [:div.settings__header
   [:h1.settings__header-headline
    "Settings"]
   [:div.settings__header-description
    "Change your personal settings"]])

(rum/defc content < rum/static
  "Render the content of the settings page."
  [system page]
  (layout
   system page
   [:div.settings__content
    (headline)
    ;; (geo-location system page)
    (units system page)]))

(rum/defcs page < (coolant/mixin settings/page)
  "Render the settings page."
  [page system]
  (content system page))

(defmethod system/change-route :settings [system route]
  (system/route-changed system route)
  (when-not (system/logged-in? system)
    (system/navigate-to-url system "/signin")))

(defmethod render-server :settings [system]
  #?(:clj (->> (coolant/get system settings/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :settings)
