(ns burningswell.web.modules.welcome
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.welcome :as welcome]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.resources :as resources]
            [burningswell.web.system.core :as system]
            [burningswell.web.ui.country-list :as country-list]
            [burningswell.web.ui.layout :as layout]
            [burningswell.web.ui.links :as link-to]
            [burningswell.web.ui.region-list :as region-list]
            [burningswell.web.ui.spot-list :as spot-list]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn query-params
  "Return the query params for the spots page."
  [system & [params]]
  (merge (system/current-location-params system)
         (api/pagination system :welcome)
         params))

(defn load-request
  [system key api-fn & args]
  (go (let [response (<! (apply api-fn system args))]
        (system/dispatch! system key response))))

(defn load-data
  "Fetch the welcome data."
  [system & [params]]
  (let [params (query-params system)]
    (go [(<! (load-request system :welcome/spots api/spots params))
         (<! (load-request system :welcome/regions api/regions (assoc params :min-spots 1)))
         (<! (load-request system :welcome/countries api/countries (assoc params :min-spots 1)))])))

(rum/defc nearby-list < rum/static
  [system title path collection]
  [:div.nearby-list
   [:div.nearby-list__header
    [:h4.nearby-list__title title]
    [:div.nearby-list__more
     (link-to/more system path)]]
   [:div.nearby-list__content collection]])

(defn country-list [system page]
  (->> (country-list/country-list system (:countries page))
       (nearby-list system "Nearby Countries" "/countries")))

(defn region-list [system page]
  (->> (region-list/region-list system (:regions page))
       (nearby-list system "Nearby Regions" "/regions")))

(defn spot-list [system page]
  (->> (spot-list/spot-list system (:spots page))
       (nearby-list system "Nearby Spots" "/spots")))

(rum/defc content < rum/static
  "Render the content of the spots page."
  [system {:keys [spots] :as page}]
  (layout/layout
   system page
   [:div.welcome__content
    (spot-list system page)
    (region-list system page)
    (country-list system page)]))

(rum/defcs page < (coolant/mixin welcome/page)
  "Render the welcome page."
  [page system]
  (content system page))

(defmethod system/change-route :welcome [system route]
  (system/route-changed system route)
  (load-data system (:query-params route)))

(defmethod render-server :welcome [system]
  #?(:clj (->> (coolant/get system welcome/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :welcome)
