(ns burningswell.web.application
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go-loop]]))
  (:require #?(:clj [clojure.core.async :refer [<! go-loop]])
            #?(:cljs [burningswell.web.modules :as modules])
            #?(:cljs [goog.dom :as gdom])
            [burningswell.web.system.bus :as bus]
            [burningswell.web.system.event-handler :as event-handler]
            [burningswell.web.system.geo-location :as geo-location]
            [burningswell.web.logging :as log]
            [burningswell.web.modules.root :as root]
            [burningswell.web.system.router :as router]
            [burningswell.web.system.core :as system]
            [burningswell.web.util :as util]
            [com.stuartsierra.component :as component]
            [rum.core :as rum]))

(def dependencies
  "The dependencies of the application component."
  [:api-client :bus :config :coolant :event-handler :geo-location :history
   :local-storage :router :service-worker])

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.application"))

(defn root-element-id
  "Return the id of the root element."
  [application]
  (-> application :config :elements :root))

(defn root-element
  "Return the DOM element of the system."
  [application]
  #?(:cljs (js/document.getElementById (root-element-id application))))

(defn mounted?
  "Returns true if `application` has been mounted, otherwise false."
  [application]
  (-> application :state deref :mounted?))

(defn mount
  "Mount the `application` using `component`."
  [application component]
  #?(:cljs
     (when-not (mounted? application)
       (swap! (:state application) assoc :mounted? true)
       (rum/mount
        (component application)
        (root-element application)))))

(defn unmount
  "Unmount the `application`."
  [application]
  (swap! (:state application) assoc :mounted? false)
  #?(:cljs (rum/unmount (root-element application))))

(defn remount
  "Remount `component`."
  [application component]
  (unmount application)
  (mount application component))

(defn- load-module-for-route
  "Load the module for `route`."
  [application route]
  #?(:cljs (modules/load-module
            (:name route)
            (fn [_]
              ;; (system/route-changed application route)
              (system/change-route application route)
              (mount application root/page)))))

(defn- handle-navigation
  "Handle navigation events."
  [{:keys [bus] :as application}]
  (let [chan (bus/subscribe bus ::router/navigate)]
    (go-loop []
      (when-let [{:keys [route]} (<! chan)]
        (load-module-for-route application route)
        (recur)))))

(defn- handle-geo-location
  "Handle geo location events."
  [{:keys [bus] :as application}]
  (let [chan (bus/subscribe bus ::geo-location/location)]
    (go-loop []
      (when-let [{:keys [location]} (<! chan)]
        (system/change-current-location application location)
        (recur)))))

(defn- handle-viewport-resize
  "Handle resize events."
  [{:keys [event-handler router] :as application}]
  (let [current-width #(:width (util/viewport-size))
        previous-width (atom (current-width))]
    (system/viewport-size-changed application (util/viewport-size))
    #?(:cljs (->> #(do (system/viewport-size-changed application (util/viewport-size))
                       (when-not (= @previous-width (current-width))
                         ;; TODO: Needed?
                         ;; (router/reload! router)
                         (reset! previous-width (current-width))))
                  (event-handler/listen event-handler js/window :resize)))))

(defrecord Application [state]
  component/Lifecycle
  (start [application]
    (system/login application)
    (handle-viewport-resize application)
    (handle-navigation application)
    (handle-geo-location application)
    (log/info logger "Application started.")
    application)
  (stop [application]
    (unmount application)
    (log/info logger "Application stopped.")
    application))

(defn application
  "Return a new event handler component."
  [& [opts]]
  (-> {:state (atom {:mounted? false})}
      (merge opts)
      (map->Application)
      (component/using dependencies)))
