(ns burningswell.web.modules.regions
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.regions :as regions]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.regions :as store]
            [burningswell.web.system.core :as system]
            [burningswell.web.ui.layout :as layout]
            [burningswell.web.ui.region-list :refer [infinite-region-list]]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [rum.core :as rum]))

(defn regions-params
  "Return the query params for the regions page."
  [system & [params]]
  (merge {:min-spots 1}
         (system/current-location-params system)
         (api/pagination system :collection)
         params))

(defn load-regions
  "Fetch the regions."
  [system & [params]]
  (let [params (regions-params system params)]
    (system/dispatch! system :regions/load-regions-start params)
    (go (let [{:keys [status body]} (<! (api/regions system params))]
          (case status
            200 (system/dispatch! system :regions/load-regions-success body)
            (system/dispatch! system :regions/load-regions-fail body))))))

(defn load-next [system]
  (let [page (coolant/get system regions/page)]
    (when-not (:loading? page)
      (load-regions system {:page (inc (:page page))}))))

(rum/defc content < rum/static
  "Render the content of the regions page."
  [system {:keys [regions] :as page}]
  (layout/layout
   system page
   [:div.regions__content
    (infinite-region-list
     system regions
     {:on-load #(load-next system)
      :scroll-target layout/scroll-target})]))

(rum/defcs page < (coolant/mixin regions/page)
  "Render the regions page."
  [page system]
  (content system page))

(defmethod system/change-route :regions [system route]
  (coolant/register-stores! system [store/store])
  (system/route-changed system route)
  (load-regions system (:query-params route)))

(defmethod render-server :regions [system]
  #?(:clj (->> (coolant/get system regions/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :regions)
