(ns burningswell.web.modules.spot
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [#?(:clj clj-time.core :cljs cljs-time.core) :as t]
            [#?(:clj clj-time.coerce :cljs cljs-time.coerce) :as coerce]
            [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.spot :as spot]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.spot :as store]
            [burningswell.web.system.core :as system]
            [burningswell.web.time :as time] 
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.spot-header :refer [spot-header]]
            [burningswell.web.ui.spot-list :refer [spot-list]]
            [burningswell.web.ui.weather.table :refer [weather-table]]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [geo.core :refer [distance-to]]
            [rum.core :as rum]))

(defn spot-weather [system spot & [{:keys [start end]}]]
  (go (let [start (or start (time/today))
            end (or end (time/plus start (time/days 1)))
            units (some-> system system/current-user :_embedded :settings :units)
            {:keys [status body]} (<! (api/spot-weather
                                       system spot
                                       {:start (time/query-param start)
                                        :end (time/query-param end)
                                        :units units}))]
        (system/dispatch! system :spot/weather [spot body])
        (system/dispatch! system :spot/time-range [spot start end]))))

(defn spot
  "Fetch a spot."
  [system spot & [opts]]
  (let [start (or (:start opts) (time/today))
        end (or (:end opts) (time/plus start (time/days 1)))]
    (system/dispatch! system :spot/spot spot)
    (go (system/dispatch!
         system
         :spot/spot
         (:body (<! (api/spot system spot (api/pagination system :collection)))))
        (system/dispatch!
         system
         :spot/photos
         [spot (:body (<! (api/spot-photos system spot)))])
        (spot-weather system spot opts)
        (system/dispatch!
         system
         :spot/spots
         [spot (:body (<! (api/spots-around system spot (api/pagination system :spots-around))))])
        ;; (wave-heights-charts system (conj spots spot))
        (system/dispatch! system :page/loading false))))

(defn distance-between
  "Calculate the distance in km between `spot-1` and `spot-2`."
  [spot-1 spot-2]
  (distance-to (:location spot-1) (:location spot-2)))

(defn assoc-distance-between
  "Assoc onto `spots` the distance in km between them and `spot`."
  [spot spots]
  (map #(assoc % :distance (distance-between spot %)) spots))

(defn previous-weather
  [system spot {:keys [start end]}]
  (fn [event]
    (spot-weather system spot
                  {:start (t/minus (coerce/to-date-time start) (t/days 1))
                   :end start})))


(defn next-weather
  [system spot {:keys [start end]}]
  (fn [event]
    (spot-weather system spot
                  {:start end
                   :end (t/plus (coerce/to-date-time end) (t/days 1))})))

(rum/defc spots-around < rum/static
  "Render the `spots` that are around `spot`."
  [system spot spots]
  (when-not (empty? spots)
    [:div.spot__spots-around
     [:h2.spot__spots-around-headline "Nearby Spots"]
     (spot-list system (assoc-distance-between spot spots))]))

(rum/defc weather-card < rum/static
  "Render the weather forecast for `spot`."
  [system spot time-range weather]
  (when-not (empty? weather)
    [:div.spot__weather
     [:h2.spot__weather-headline "Surf Forecast"]
     (weather-table
      weather
      {:on-previous (previous-weather system spot time-range)
       :on-next (next-weather system spot time-range)})]))

(rum/defc content < rum/static
  "Render the content of the spot page."
  [system {:keys [spot spots photos time-range weather] :as page}]
  (layout
   system page
   [:div.spot__content
    (spot-header system spot photos)
    [:div.spot__details
     (weather-card system spot time-range weather)
     (spots-around system spot spots)]]))

(rum/defcs page < (coolant/mixin spot/page)
  "Render the spot page."
  [page system]
  (content system page))

(defmethod system/change-route :spot [system route]
  (coolant/register-stores! system [store/store])
  (system/route-changed system route)
  (spot system (:params route)))

(defmethod render-server :spot [system]
  #?(:clj (->> (coolant/get system spot/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :spot)
