(ns burningswell.web.ui.search-input
  (:require [burningswell.web.system.core :as system]
            [burningswell.web.system.history :as history]
            [burningswell.web.ui.auto-complete :refer [auto-complete]]
            [burningswell.web.util :as util]
            [burningswell.web.system.router :refer [routes]]
            [routes.core :refer [path-for]]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn search-result-key
  "Return the React key for a region list item."
  [result]
  (str "search-result-" (:type result) "-" (:id result)))

(rum/defc country < rum/static {:key-fn search-result-key}
  "Render a country search result."
  [result]
  (let [{:keys [continent country]} (:_embedded result)]
    [:div.auto-complete-item
     [:div.auto-complete-item__headline
      (util/country-flag country)
      (:term result)]
     [:div.auto-complete-item__address
      (mdl/icon "location_on")
      (:name continent)]]))

(rum/defc region < rum/static {:key-fn search-result-key}
  "Render a region search result."
  [result]
  (let [{:keys [continent country]} (:_embedded result)]
    [:div.auto-complete-item
     [:div.auto-complete-item__headline
      (util/country-flag country)
      (:term result)]
     [:div.auto-complete-item__address
      (mdl/icon "location_on")
      (:name country) ", " (:name continent)]]))

(rum/defc spot < rum/static {:key-fn search-result-key}
  "Render a spot search result."
  [result]
  (let [{:keys [continent country region]} (:_embedded result)]
    [:div.auto-complete-item
     [:div.auto-complete-item__headline
      (util/country-flag country)
      (:term result)]
     [:div.auto-complete-item__address
      (mdl/icon "location_on")
      (if region (:name region))
      (if region ", ")
      (:name country) ", " (:name continent)]]))

(rum/defc result < rum/static
  "Render a search result."
  [result]
  (case (:type result)
    :country (country result)
    :region (region result)
    :spot (spot result)))

(defn- on-change
  "Update the search query in the app state."
  [system query cancel]
  #?(:cljs (system/auto-complete system query cancel {:per-page 8})))

(defn- search-path
  "Return the relative path of `result`."
  [query]
  (->> {:query query
        :page 1
        :per-page 24}
       (path-for routes :search)))

(defn- search-result-path
  "Return the relative path of `result`."
  [result]
  (path-for routes (:type result) result))

(defn- on-select
  "Handle a selected search result."
  [system result cancel]
  (on-change system (:term result) cancel)
  (history/set-token! (:history system) (search-result-path result)))

(defn- on-submit [system query cancel]
  (history/set-token! (:history system) (search-path query)))

(rum/defc search-input < rum/static
  "Render the search input field."
  [system {:keys [id cancel class query results]}]
  (auto-complete
   query results
   {:id id
    :class class
    :on-change #(on-change system % cancel)
    :on-select #(on-select system % cancel)
    :on-submit #(on-submit system % cancel)
    :placeholder "Search Burning Swell"
    :render result}))
