(ns burningswell.web.modules.map
  (:require [burningswell.web.actions :as actions]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.map :as map]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.ui.google-map :as google-map]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.search-list :refer [search-list]]
            [rum.core :as rum]))

(defn map-location
  "Return the current map location."
  [system page]
  (or (-> page :map :center)
      (-> system :local-storage :map-center)
      (-> system :local-storage :location)))

(defn map-zoom
  "Return the current map zoom level."
  [system page]
  (or (-> page :map :zoom)
      (-> system :local-storage :map-zoom)))

(defn on-map-idle
  "Handle map idle events."
  [system page map]
  (actions/change-map-center system (google-map/center map))
  (actions/change-map-bounding-box system (google-map/bounding-box map))
  (actions/map-idle
   system
   (google-map/bounding-box map)
   (google-map/center map)
   (google-map/zoom map)))

(defn on-center-changed
  "Handle center change events."
  [system page map]
  (actions/change-map-center system (google-map/center map)))

(defn on-zoom-changed
  "Handle zoom change events."
  [system page map]
  (actions/change-map-zoom system (google-map/zoom map)))

(rum/defc content
  "Render the content of the map page."
  [system {:keys [results] :as page}]
  (layout
   system page
   [:div.map__content
    (google-map/google-map
     {:class "map__map"
      :location (map-location system page)
      :zoom (map-zoom system page)
      :on-idle #(on-map-idle system page %)
      :on-zoom-changed #(on-zoom-changed system page %)})
    (search-list system results)]))

(rum/defcs page < (coolant/mixin map/page)
  "Render the map page."
  [page system]
  (content system page))

(defmethod render-server :map [system]
  #?(:clj (->> (coolant/get system map/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :map)
