(ns burningswell.web.history
  (:require [burningswell.web.bus :as bus]
            [burningswell.web.logging :as log]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            #?(:cljs [goog.events :as events])
            #?(:cljs [goog.history.EventType :as EventType]))
  #?(:cljs (:import goog.History goog.history.Html5History)))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.history"))

(defn- normalize-path
  "Return a normalized token."
  [path]
  (str/replace path #"^/" ""))

(defn remove-query-params
  "Remove the query params in `url`."
  [url]
  (if-let [index (str/index-of url "?")]
    (subs url 0 index) url))

(defn remove-query-params!
  "Remove the query params from the current url."
  [history]
  #?(:cljs (let [url js/window.location.href]
             (js/window.history.replaceState
              #js {} js/document.title
              (remove-query-params url)))))

(defn set-token!
  "Set the history token."
  [history path]
  (remove-query-params! history)
  (.setToken (:backend history) (normalize-path path)))

(defn replace-token!
  "Replace the history token."
  [history path]
  (remove-query-params! history)
  (.replaceToken (:backend history) (normalize-path path)))

(defn replace-state!
  "Replace the history token without firing a history event."
  [history path]
  (remove-query-params! history)
  #?(:cljs (js/window.history.replaceState #js {} js/document.title (normalize-path path))))

(defn listen
  "Listen for EventType/NAVIGATE events on `history` and publish them
  on the message bus."
  [history]
  (->> #?(:clj nil
          :cljs (events/listen
                 (:backend history)
                 EventType/NAVIGATE
                 (fn [event]
                   (let [msg {:token (.-token event)}]
                     (bus/publish (:bus history) :history/navigate msg)
                     (log/debug logger (str "History event published. " msg))))))
       (assoc history :listener)))

(defn select-backend
  "Return the history backend."
  [history]
  (->> #?(:clj nil
          :cljs (if (goog.history.Html5History.isSupported)
                  (doto (goog.history.Html5History.)
                    (.setUseFragment false))
                  (goog.History.)))
       (assoc history :backend)))

(defn start-history
  "Start the history."
  [history]
  (if (:backend history)
    history
    (let [history (select-backend history)
          history (listen history)]
      #?(:cljs (.setEnabled (:backend history) true))
      (log/info logger "History successfully started.")
      history)))

(defn stop-history
  "Stop the history."
  [history]
  (when-let [backend (:backend history)]
    #?(:cljs (.setEnabled backend false)))
  (when-let [listener (:listener history)]
    #?(:cljs (events/unlistenByKey listener)))
  (log/info logger "History successfully stopped.")
  (assoc history :backend nil :listener nil))

(defrecord History [backend fragment listener]
  component/Lifecycle
  (start [history]
    (start-history history))
  (stop [history]
    (stop-history history)))

(defn history
  "Return a new history component."
  [& [opts]]
  (component/using
   (map->History opts)
   [:bus]))
