(ns burningswell.web.errors
  (:require [clojure.string :as str]
            [geo.core :as geo]))

(defmulti message-for
  "Return the error message for `key`."
  (fn [key data] key))

(defmethod message-for :email
  [key {:keys [error value]}]
  (cond
    (str/blank? value)
    "Email can't be blank."
    (= (-> error rest ffirst) 'email?)
    "Email is not valid."
    (= error `(~'not (~'email-available? ~value)))
    "Email has already been taken."))

(defmethod message-for :latitude
  [key {:keys [error value]}]
  (cond
    (js/isNaN value)
    "Not a number."
    (not (geo/latitude? value))
    "Invalid latitude."
    :else nil))

(defmethod message-for :longitude
  [key {:keys [error value]}]
  (cond
    (js/isNaN value)
    "Not a number."
    (not (geo/longitude? value))
    "Invalid longitude."
    :else nil))

(defmethod message-for :login
  [key {:keys [error value]}]
  (if (str/blank? value)
    "Login can't be blank."))

(defmethod message-for :password
  [key {:keys [error value]}]
  (cond
    (str/blank? value)
    "Password can't be blank."
    (= error :incorrect-credentials)
    "Incorrect credentials. Try again!"
    (= (-> error rest ffirst) 'min-size?)
    (str "Password is too short (min. "
         (-> error rest first second) " chars).")
    (= (-> error rest ffirst) 'max-size?)
    (str "Password is too long (max. "
         (-> error rest first second) " chars).")))

(defmethod message-for :username
  [key {:keys [error value]}]
  (cond
    (str/blank? value)
    "Username can't be blank."
    (= error `(~'not (~'username-available? ~value)))
    "Username has already been taken."
    (= (-> error rest ffirst) 'min-size?)
    (str "Username is too short (min. "
         (-> error rest first second) " chars).")
    (= (-> error rest ffirst) 'max-size?)
    (str "Username is too long (max. "
         (-> error rest first second) " chars).")))

(defmethod message-for :terms-of-service
  [key {:keys [error value]}]
  (if (and (not (true? value)) error)
    "Please accept the Terms of Service."))

(defmethod message-for :new-spot/name
  [key {:keys [error value]}]
  (cond
    (str/blank? value)
    "Name can't be blank."
    (= (-> error rest ffirst) 'min-size?)
    (str "Name is too short (min. "
         (-> error rest first second) " chars).")
    (= (-> error rest ffirst) 'max-size?)
    (str "Name is too long (max. "
         (-> error rest first second) " chars).")
    :else nil))

(defmethod message-for :session/rating
  [key {:keys [error value]}]
  (cond
    (= error '(not (integer? nil)))
    (str "Please select 1 of the 5 stars.")
    :else nil))
