(ns burningswell.web.authentication
  (:require [apollo.core :as a]
            [burningswell.web.browser :as browser]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.history :as history]
            [burningswell.web.logging :as log]
            [com.stuartsierra.component :as component]))

(def logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(defn- dissoc-query-param [url query-param]
  (update url :query-params dissoc (keyword query-param)))

(defn- auth-token-name
  "Returns the name of the auth token as a string."
  [authentication]
  (:auth-token authentication))

(defn auth-token-from-url
  "Get the auth token from `url`."
  [{:keys [auth-token]} url]
  (get-in url [:query-params (keyword auth-token)]))

(defn auth-token-from-session-storage
  "Save the auth token query parameter to session storage."
  [{:keys [auth-token session-storage]}]
  (get session-storage (keyword auth-token)))

(defn clear-auth-token-from-cookies!
  "Clear the auth token from the cookies."
  [authentication]
  (cookies/remove-cookie! (auth-token-name authentication)))

(defn clear-auth-token-from-storage!
  "Clear the auth token from local and session storage."
  [{:keys [auth-token local-storage session-storage]}]
  (let [key (keyword auth-token)]
    (dissoc! local-storage key)
    (dissoc! session-storage key)
    (log/info logger "Removed authentication token from storage.")))

(defn clear-auth-token!
  "Clear the auth token from cookies and storage."
  [authentication]
  (clear-auth-token-from-cookies! authentication)
  (clear-auth-token-from-storage! authentication))

(defn- set-clean-browser-location!
  [{:keys [auth-token history]} url]
  (when history
    (some->> (dissoc-query-param (browser/url) auth-token)
             (history/set-url! history))
    (log/info logger "Removed authentication token from browser location.")))

(defn save-auth-token-to-cookie!
  "Save the `auth-token` to a cookie."
  [authentication auth-token]
  (cookies/set-cookie!
   (auth-token-name authentication)
   (str auth-token)
   {:path "/"
    :secure false ;; TODO: True in production, otherwise false.
    }))

(defn save-auth-token-to-session-storage!
  "Save the `auth-token` to session storage."
  [{:keys [session-storage] :as authentication} auth-token]
  (when session-storage
    (assoc! session-storage (keyword (auth-token-name authentication)) auth-token)
    (log/info logger "Saved authentication token to session storage.")))

(defn save-auth-token!
  "Save the `auth-token` to session storage."
  [authentication auth-token]
  (save-auth-token-to-cookie! authentication auth-token)
  (save-auth-token-to-session-storage! authentication auth-token))

(defn- handle-browser-location
  "Handle login from the auth token query parameter in the browser location"
  [{:keys [client] :as authentication}]
  (when-let [url (browser/url)]
    (when-let [auth-token (auth-token-from-url authentication url)]
      (set-clean-browser-location! authentication url)
      (save-auth-token! authentication auth-token)
      (log/info logger "Successfully logged in.")
      (a/on-reset-store client identity))))

(defrecord Authentication [history]
  component/Lifecycle
  (start [this]
    (handle-browser-location this)
    (log/info logger "Authentication started.")
    this)
  (stop [this]
    (log/info logger "Authentication stopped.")
    this))

(defn authentication []
  (-> (map->Authentication {:auth-token "auth-token"})
      (component/using [:client :history :local-storage :session-storage])))

(defn ssr-authentication []
  (-> (map->Authentication {:auth-token "auth-token"})
      (component/using [:local-storage :session-storage])))
