(ns burningswell.web.server.routes
  (:require ["react-dom/server" :as react-dom-server]
            [bidi.bidi :as bidi]
            [burningswell.web.client :as client]
            [burningswell.web.core :as web]
            [burningswell.web.main :as main]
            [burningswell.web.pages.continent]
            [burningswell.web.pages.continents]
            [burningswell.web.pages.countries]
            [burningswell.web.pages.country]
            [burningswell.web.pages.reset-password]
            [burningswell.web.pages.home]
            [burningswell.web.pages.map]
            [burningswell.web.pages.region]
            [burningswell.web.pages.regions]
            [burningswell.web.pages.search]
            [burningswell.web.pages.signin]
            [burningswell.web.pages.signup]
            [burningswell.web.pages.spot]
            [burningswell.web.pages.spots]
            [burningswell.web.pages.users]
            [burningswell.web.router :as router]
            [burningswell.web.storage :as storage]
            [burningswell.web.system :as system]
            [burningswell.web.viewport :as viewport]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [hodgepodge.core :as hodgepodge]
            [macchiato.util.response :as r]
            [react-apollo :as react-apollo]
            [sablono.core :refer [defhtml html]]))

(def seo-description
  "The SEO descrition."
  "Surf Spot Weather Forecasts")

(def seo-keywords
  "The SEO keywords."
  ["Surf Report"
   "Surf Forecast"
   "Surf"
   "Surfing"])

(defn render-state [state]
  (let [text (js/JSON.stringify state)]
    ;; TODO: escape state
    (html [:script {:dangerouslySetInnerHTML {:__html (str "window.__APOLLO_STATE__= " text ";")}}])))

(defn- public-config
  "Returns the public `config` of the web application."
  [{:keys [api-client google] :as config}]
  {:api-client (select-keys api-client [:scheme :server-name :server-port :uri])
   :google {:maps (select-keys (:maps google) [:api-key])}})

(defhtml include-js-async
  "Include a list of external javascript files."
  [script async?]
  [:script
   {:async (true? async?)
    :type "text/javascript"
    :src script}])

(defn include-google-maps
  "Include the JavaScript for Google Maps."
  [config]
  (include-js-async
   (str "https://maps.googleapis.com/maps/api/js?key="
        (-> config :google :maps :api-key))
   true))

(defn document
  "Render the HTML document."
  [state config content]
  (let [main-opts (js/JSON.stringify (clj->js (public-config config)))]
    (html
     [:html
      [:head
       [:meta {:char-set "UTF-8"}]
       [:meta {:name "description" :content seo-description}]
       [:meta {:name "keywords" :content (str/join ", " seo-keywords)}]
       [:meta {:name "viewport" :content "width=device-width, initial-scale=1.0"}]
       [:link {:rel "stylesheet" :href "https://fonts.googleapis.com/css?family=Roboto:300,400,500"}]
       [:link {:rel "stylesheet" :href "https://fonts.googleapis.com/icon?family=Material+Icons"}]
       [:link {:rel "stylesheet" :href "/stylesheets/burningswell.css"}]]
      [:body
       [:div#app {:dangerouslySetInnerHTML {:__html content}}]
       (render-state state)
       (include-google-maps config)
       [:script {:src "/javascripts/main.js"}]
       [:script
        {:dangerouslySetInnerHTML
         {:__html (str "window.onload = function() {"
                       "  burningswell.web.main.main(" main-opts ");"
                       "};")}}]]])))

(defn- clear-storage!
  "Clear the local and session storage."
  []
  (hodgepodge/clear! hodgepodge/local-storage)
  (hodgepodge/clear! hodgepodge/session-storage))

(defn- setup-storage!
  "Setup the local and session storage."
  [request]
  (clear-storage!)
  (some->> (get-in (:cookies request) ["auth-token" :value])
           (assoc! hodgepodge/session-storage :auth-token)))

(defn render-error
  "Render an error page."
  [error]
  (html [:div
         [:h1 "Internal Server Error"]
         [:pre (with-out-str
                 (prn error)
                 (.log js/console error))]]))

(defn- shutdown-system! [system]
  (try (component/stop system)
       (catch js/Error e
         (println "Can't shutdown system.")
         (.log js/console e))))

(defn- cookie-location [request]
  (let [latitude (get-in request [:cookies "latitude" :value])
        longitude (get-in request [:cookies "longitude" :value])]
    (when (and latitude longitude)
      {:latitude (js/parseFloat latitude)
       :longitude (js/parseFloat longitude)})))

(defn- initial-state [request]
  {:location (cookie-location request)
   :request (select-keys request [:scheme :server-name :server-port :uri :query-params])
   :viewport (viewport/cookies request)})

(defn render [server req res raise]
  (setup-storage! req)
  (let [config (:config server)
        system (system/ssr-system config (initial-state req))
        system (component/start system)
        page (main/view system (:route req))]
    (-> (react-apollo/getDataFromTree page)
        (.then (fn []
                 (let [content (react-dom-server/renderToString page)
                       state (.extract (:client system))]
                   (shutdown-system! system)
                   (-> (r/ok (react-dom-server/renderToStaticMarkup (document state config content)))
                       (r/content-type "text/html")
                       (res)))))
        (.catch (fn [error]
                  (shutdown-system! system)
                  (-> (r/ok (react-dom-server/renderToStaticMarkup (render-error error)))
                      (r/content-type "text/html")
                      (res)))))))

(defn not-found [req res raise]
  (-> (html
       [:html
        [:body
         [:h2 (:uri req) " was not found"]]])
      (react-dom-server/renderToString)
      (r/not-found)
      (r/content-type "text/html")
      (res)))

(defn- match-route [{:keys [uri]}]
  (when-let [route (bidi/match-route router/routes uri)]
    (assoc route :uri uri :loaded true)))

(defn router [server]
  (fn [req res raise]
    (if-let [route (match-route req)]
      (render server (assoc req :route route route) res raise)
      (not-found req res raise))))
