(ns burningswell.web.ui.auto-complete
  (:require [apollo.core :as a]
            [burningswell.web.ui.auto-complete.continents :as continents]
            [burningswell.web.ui.auto-complete.countries :as countries]
            [burningswell.web.ui.auto-complete.regions :as regions]
            [burningswell.web.ui.auto-complete.spots :as spots]
            [burningswell.web.ui.icon :refer [icon]]
            [goog.events.KeyCodes :as KeyCodes]
            [sablono.core :refer [defhtml html]]))

(def default-data
  {:__typename "AutoComplete"
   :id nil
   :index -1
   :load_details false
   :show_results false
   :query ""})

(a/defgql state-query
  '((query
     AutoComplete
     [($id String)]
     (auto-complete
      [(id $id)]
      ((client))
      id
      index
      load-details
      show-results
      query
      __typename))))

(a/defgql search-query
  `((query
     AutoComplete
     [($query String)]
     (search
      [(query $query)]
      (edges
       (node
        (... continent)
        (... country)
        (... region)
        (... spot)))))
    ~continents/fragment
    ~countries/fragment
    ~regions/fragment
    ~spots/fragment))

(defn resolve-data [obj args context info]
  (if-let [id (:id args)]
    (assoc default-data :id id)
    default-data))

(defn- read-state [client form]
  (a/read-query client state-query {:variables {:id (:id form)}}))

(defn- update-state! [client form f & args]
  (let [state (read-state client form)]
    (a/write-data! client (apply f state args))))

(defn- search-results
  "Returns the search results of `component`."
  [data]
  (a/nodes data :search))

(defn- set-index!
  "Set the auto complete index."
  [client form index]
  (update-state! client form assoc-in [:auto_complete :index] index))

(defn- set-show-results!
  "Set the :show-results? flag."
  [client form show-results?]
  (update-state! client form assoc-in [:auto_complete :show_results] show-results?))

(defn on-clear-query
  "Clear the current query."
  [client form]
  (set-index! client form -1)
  (set-show-results! client form false)
  (update-state! client form assoc-in [:auto_complete :query] "")
  (when-let [handler (:on-change form)]
    (handler nil)))

;; (defn- default-select
;;   "The default selection handler."
;;   [component selection]
;;   (some->> selection :slug :path
;;            (history/set-token! (a/env component :history))))

(defn on-select
  "Handle selecting a search result."
  [client form data]
  (set-show-results! client form false)
  (let [index (:index form)
        selection (nth (search-results data) index)]
    (if-let [handler (:on-select form)]
      (handler selection)
      ;; (default-select component selection)
      )))

(defn on-submit
  "Handle submitting a query."
  [form]
  (when-let [handler (:on-submit form)]
    (handler (:query form))))

(defn on-enter-key
  "Handle enter key."
  [client form data]
  (set-show-results! client form false)
  (update-state! client form assoc-in [:auto_complete :load-details] true)
  (let [index (:index form)]
    (if (neg? (:index form))
      (on-submit form)
      (on-select client form data))))

(defn on-escape-key
  "Handle escape key."
  [client form]
  (set-show-results! client form false)
  (update-state! client form assoc-in [:auto_complete :load_details] false))

(defn- select-next
  "Move to the next auto complete result."
  [client form data]
  (let [next (inc (:index form))
        results (search-results data)]
    ;; (prn "RESULTS" (-> form :auto_complete :index) next (count results))
    ;; (set-show-results! client form true)
    (if (< next (count results))
      (set-index! client form next)
      (set-index! client form -1))))

(defn- select-previous
  "Move to the previous auto complete result."
  [client form data]
  (let [prev (dec (:index form))
        results (search-results data)]
    ;; (set-show-results! client form true)
    (if (= prev -2)
      (set-index! client form (dec (count results)))
      (set-index! client form prev))))

(defn- on-query-change
  "Handle query changes."
  [client form event]
  (let [value (.. event -target -value)]
    (update-state! client form assoc-in [:auto_complete :query] value)
    (set-index! client form -1)
    (set-show-results! client form true)
    (when-let [handler (:on-change form)]
      (handler value))))

(defn on-key-down
  "Handle key down events."
  [client form data event]
  (condp = (.-keyCode event)
    KeyCodes/DOWN
    (do (.preventDefault event)
        (select-next client form data))
    KeyCodes/ENTER
    (on-enter-key client form data)
    KeyCodes/ESC
    (on-escape-key client form)
    KeyCodes/UP
    (do (.preventDefault event)
        (select-previous client form data))
    nil))

(defn clear-button
  "Render the clear query button."
  [client form]
  (icon "clear" {:class "auto-complete__clear"
                 :on-click #(on-clear-query client form)}))

(defn search-icon
  "Render the search icon."
  []
  [:div.auto-complete__icon-search
   {} (icon "search")])

(defn loading-icon
  "Render the loading icon."
  []
  ;; (if load-details?
  ;;   "Loading" ;; (mdl/spinner {:class "auto-complete__loading"})
  ;;   (core/icon "search"))
  )

(defn query-input
  "Render the query input."
  [client form data & [{:keys [auto-focus placeholder]}]]
  (let [{:keys [auto-focus placeholder]} form]
    (html [:input.auto-complete__query
           {:auto-focus (true? auto-focus)
            :on-change #(on-query-change client form %)
            :on-click #(set-show-results! client form true)
            :on-key-down #(on-key-down client form data %)
            :placeholder (or placeholder "Search Burning Swell")
            :type "text"
            :value (:query form)}])))

(defmulti search-result
  "Render a search result."
  (fn [result] (-> result :__typename keyword)))

(defmethod search-result :Continent [continent]
  (continents/suggestion continent))

(defmethod search-result :Country [country]
  (countries/suggestion country))

(defmethod search-result :Region [region]
  (regions/suggestion region))

(defmethod search-result :Spot [spot]
  (spots/suggestion spot))

(defhtml suggestions
  "Render the suggestions."
  [form data]
  (let [{:keys [index show_results]} form
        results (search-results data)]
    [:div.auto-complete__suggestions
     {:class (if (or (empty? results)
                     (not show_results))
               "auto-complete__suggestions--hidden")}
     (for [[current result] (map-indexed vector results)]
       [:div.auto-complete__result
        {:class (if (= current index) "auto-complete__result--highlight")
         :key (str "auto-complete-item-" (:id result))}
        (search-result result)])]))

(defn auto-complete
  "Render an auto complete component."
  [{:keys [id class] :as form}]
  (a/with-query [{:keys [data]}]
    {:query state-query :variables {:id id}}
    (let [form (merge (:auto_complete data) form)]
      (a/with-query [{:keys [client data error loading]}]
        {:key "auto-complete"
         :query search-query
         :variables {:query (:query form)}}
        (html
         [:div.auto-complete
          {:class class}
          (search-icon)
          (loading-icon)
          (query-input client form data)
          (clear-button client form)
          (suggestions form data)])))))
