(ns burningswell.web.ui.lists
  (:require [apollo.core :as a]
            [burningswell.web.core :as core]
            [burningswell.web.history :as history]
            [burningswell.web.ui.links :as links]
            [burningswell.web.ui.list :as list]
            [burningswell.web.ui.system :as system]
            [cljs.loader :as loader]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [react :as React :refer [createElement]]
            [sablono.core :refer [html]]))

(defn- go-to-slug! [history node]
  (history/set-path! history (a/get node :slug :path)))

(defn- title-tag [title]
  (html [:div.mdc-typography--caption title]))

#_(defn- photo-tag [node]
    (let [photo (a/get node :photo)
          image (first (a/nodes photo :images))]
      (html [:img {:src (a/get image :url)}])))

(defn- photo-tag [photo]
  (let [image (first (a/nodes photo :images))]
    (html [:img {:src (a/get image :url)}])))

(defn divider []
  (list/divider {:tag "div"}))

;; Countries

(a/defgql countries-query
  '((query
     countries
     [($continents [ID!])
      ($location LocationInput)
      ($min_spots Int 1)
      ($sort SORT_COUNTRY)]
     (countries
      [(continents $continents)
       (location $location)
       (min_spots $min_spots)
       (sort $sort)]
      (edges
       (node
        id
        name
        (continent
         id name
         (slug path))
        (photo
         id
         (images
          [(first 1)
           (sort WIDTH)]
          (edges
           (node id url width height))))
        (slug path)))))))

(defn countries [& [opts]]
  (a/query
   {:query countries-query
    :variables (merge {:min_spots 1} opts)}
   (fn [{:keys [client data error loading] :as props}]
     (system/consumer
      (fn [{:keys [history]}]
        (html
         [:div.country-list
          (cond
            error
            [:div "Error :("]
            loading
            [:div "Loading ..."]
            :else
            [:div
             (title-tag "Countries")
             (list/list
              {:className "country-list__list mdc-list--two-line"}
              (for [node (a/nodes data :countries)]
                (list/item
                 {:className "country-list__item"
                  :key (str "country-list-item-" (a/get node :id))
                  :onClick #(go-to-slug! history node)}
                 (list/item-graphic
                  {:graphic (photo-tag (a/get node :photo))})
                 (list/item-text
                  {:primaryText (a/get node :name)
                   :secondaryText (links/continent (:continent node))}))))])]))))))

;; Continents

(a/defgql continents-query
  '((query
     continents
     [($location LocationInput)
      ($sort SORT_CONTINENT)]
     (continents
      [(location $location)
       (sort $sort)]
      (edges
       (node
        id
        name
        (slug path)
        (photos
         [(first 1)]
         (edges
          (node
           id
           (images
            [(first 1)
             (sort WIDTH)]
            (edges
             (node id url width height))))))))))))

(defn continents [& [opts]]
  (a/query
   {:query continents-query
    :variables opts}
   (fn [{:keys [client data error loading] :as props}]
     (system/consumer
      (fn [{:keys [history]}]
        (html
         [:div.continent-list
          (cond
            error
            [:div "Error :("]
            loading
            [:div "Loading ..."]
            :else
            [:div
             (title-tag "Continents")
             (list/list
              {:className "continent-list__list mdc-list--two-line"}
              (for [node (a/nodes data :continents)
                    :let [photo (first (a/nodes node :photos))]]
                (list/item
                 {:className "continent-list__item"
                  :key (str "continent-list-item-" (a/get node :id))
                  :onClick #(go-to-slug! history node)}
                 (list/item-graphic
                  {:graphic (photo-tag photo)})
                 (list/item-text
                  {:primaryText (a/get node :name)}))))])]))))))

;; Regions

(a/defgql regions-query
  '((query
     regions
     [($countries [ID!])
      ($location LocationInput)
      ($min_spots Int 1)
      ($sort SORT_REGION)]
     (regions
      [(countries $countries)
       (location $location)
       (min_spots $min_spots)
       (sort $sort)]
      (edges
       (node
        id
        name
        (country id name)
        (photo
         id
         (images
          [(first 1)
           (sort WIDTH)]
          (edges
           (node id url width height))))
        (slug path)))))))

(defn regions [& [opts]]
  (a/query
   {:query regions-query
    :variables (merge {:min_spots 1} opts)}
   (fn [{:keys [client data error loading] :as props}]
     (system/consumer
      (fn [{:keys [history]}]
        (html
         [:div.region-list
          (cond
            error
            [:div "Error :("]
            loading
            [:div "Loading ..."]
            :else
            [:div
             (title-tag "Regions")
             (list/list
              {:className "region-list__list mdc-list--two-line"}
              (for [node (a/nodes data :regions)]
                (list/item
                 {:className "region-list__item"
                  :key (str "region-list-item-" (a/get node :id))
                  :onClick #(go-to-slug! history node)}
                 (list/item-graphic
                  {:graphic (photo-tag (a/get node :photo))})
                 (list/item-text
                  {:primaryText (a/get node :name)
                   :secondaryText (a/get node :country :name)}))))])]))))))

;; Spots

(a/defgql spots-query
  '((query
     spots
     [($countries [ID!])
      ($location LocationInput)
      ($regions [ID!])
      ($sort SORT_SPOT)]
     (spots
      [(countries $countries)
       (location $location)
       (regions $regions)
       (sort $sort)]
      (edges
       (node
        id
        name
        (country
         id name
         (slug path))
        (region
         id name
         (slug path))
        (photo
         id
         (images
          [(first 1)
           (sort WIDTH)]
          (edges
           (node id url width height))))
        (slug path)))))))

(defn spot-secondary-text
  [{:keys [country region] :as spot}]
  (html
   [:div
    (links/region region)
    (when (and country region) ", ")
    (links/country country)]))

(defn spots [& [opts]]
  (a/query
   {:query spots-query
    :variables opts}
   (fn [{:keys [client data error loading] :as props}]
     (system/consumer
      (fn [{:keys [history]}]
        (html
         [:div.spot-list
          (cond
            error
            [:div "Error :("]
            loading
            [:div "Loading ..."]
            :else
            [:div
             (title-tag "Spots")
             (list/list
              {:className "spot-list__list mdc-list--two-line"}
              (for [node (a/nodes data :spots)]
                (list/item
                 {:className "spot-list__item"
                  :key (str "spot-list-item-" (a/get node :id))
                  :onClick #(go-to-slug! history node)}
                 (list/item-graphic
                  {:graphic (photo-tag (a/get node :photo))})
                 (list/item-text
                  {:primaryText (a/get node :name)
                   :secondaryText (spot-secondary-text node)}))))])]))))))

;; Users

(a/defgql users-query
  '((query
     users
     [($location LocationInput)
      ($query String)
      ($sort SORT_USER)]
     (users
      [(location $location)
       (query $query)
       (sort $sort)]
      (edges
       (node
        id
        username
        first_name
        last_name
        (country
         id name
         (slug path))
        (region
         id name
         (slug path))
        ;; (photo
        ;;  id
        ;;  (images
        ;;   [(first 1)
        ;;    (sort WIDTH)]
        ;;   (edges
        ;;    (node id url width height))))
        (slug path)))))))

(defn user-secondary-text
  [{:keys [country region] :as user}]
  (html
   [:div
    (links/region region)
    (when (and country region) ", ")
    (links/country country)]))

(defn users [& [opts]]
  (a/query
   {:query users-query
    :variables opts}
   (fn [{:keys [client data error loading] :as props}]
     (system/consumer
      (fn [{:keys [history]}]
        (html
         [:div.user-list
          (cond
            error
            [:div "Error :("]
            loading
            [:div "Loading ..."]
            :else
            [:div
             (title-tag "Users")
             (list/list
              {:className "user-list__list mdc-list--two-line"}
              (for [node (a/nodes data :users)]
                (list/item
                 {:className "user-list__item"
                  :key (str "user-list-item-" (a/get node :id))
                  :onClick #(go-to-slug! history node)}
                 (list/item-graphic
                  {:graphic (photo-tag node)})
                 (list/item-text
                  {:primaryText (core/user-name node)
                   :secondaryText (user-secondary-text node)}))))])]))))))
