(ns burningswell.web.system
  (:require [burningswell.web.authentication :as authentication]
            [burningswell.web.viewport :as viewport]
            [burningswell.web.client :as client]
            [burningswell.web.history :as history]
            [burningswell.web.location :as location]
            [burningswell.web.logging :as logging]
            [burningswell.web.router :as router]
            [burningswell.web.session :as session]
            [burningswell.web.storage :as storage]
            [com.stuartsierra.component :as component]))

(defonce ^:dynamic *system* nil)

(defn system
  "Returns a new system using `config`."
  [{:keys [api-client] :as config}]
  (component/system-map
   :authentication (authentication/authentication)
   :client (client/browser {:uri api-client})
   :config config
   :history (history/history)
   :local-storage (storage/local-storage)
   :location (location/location)
   :logging (logging/logging)
   :router (router/router)
   :session (session/session)
   :session-storage (storage/session-storage)
   :viewport (viewport/viewport)))

(defn ssr-system
  "Returns a new server side rendering system using `config`."
  [{:keys [api-client] :as config} initial-state]
  (let [request (:request initial-state)]
    (component/system-map
     :authentication (authentication/ssr-authentication)
     :client (client/ssr {:uri api-client :defaults initial-state})
     :config config
     :local-storage (storage/local-storage)
     :location (location/location)
     :logging (logging/logging)
     :router (router/ssr-router request)
     :session (session/session)
     :session-storage (storage/session-storage))))

(defn start!
  "Start a new system using `config` and set *system*."
  [config]
  (let [started (component/start (system config))]
    (set! *system* started)
    started))

(defn stop!
  "Stop the system bound to *system*."
  []
  (some-> *system* component/stop)
  (set! *system* nil))
