(ns burningswell.web.viewport
  (:require [apollo.core :as a]
            [clojure.pprint :refer [pprint]]
            [burningswell.web.logging :as log]
            [burningswell.web.cookies :as cookies]
            [com.stuartsierra.component :as component]
            [no.en.core :refer [parse-integer]]
            [goog.dom :as dom]
            [goog.events :as events])
  (:import [goog.dom ViewportSizeMonitor]
           [goog.events EventType]))

(def fallback
  "The fallback viewport size."
  {:height 768 :width 1024})

(def logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(a/defgql query
  '((viewport ((client)) __typename height width)))

(a/defgql mutation
  `((mutation
     update-viewport
     [($height Int!)
      ($width Int!)]
     (update-viewport
      [(height $height)
       (width $width)]
      ((client))
      __typename width height))))

(defn- size->map
  "Convert the  the viewport change message."
  [size]
  {:height (.-height size)
   :width (.-width size)})

(defn viewport-size
  "Returns the viewport size of the window."
  []
  (when (exists? js/window)
    (size->map (dom/getViewportSize))))

(defn default-data
  "Returns the default viewport data."
  [& [size]]
  (assoc (or size (viewport-size)) :__typename "Viewport"))

(defn cookies
  "Returns the viewport size from the `request` cookies."
  [request]
  (let [height (get-in request [:cookies "viewport-height" :value])
        width (get-in request [:cookies "viewport-width" :value])]
    {:__typename "Viewport"
     :height (or (parse-integer height) (:height fallback))
     :width (or (parse-integer width) (:width fallback))}))

(defn update-viewport!
  "The update viewport resolver."
  [obj args context info]
  (let [data (a/read-query context query)
        data' (update data :viewport merge args)]
    (:viewport (a/write-query context query data'))))

(defn- viewport-change-msg
  "Returns the viewport change message."
  [{:keys [width height]}]
  (str "Viewport size changed to " width "x" height "."))

(defn- set-cookies!
  "Set the viewport cookies."
  [service {:keys [height width]}]
  (let [opts {:path "/" :secure false}]
    (cookies/set-cookie! :viewport-height height)
    (cookies/set-cookie! :viewport-width width)))

(defn- write-store!
  "Write the viewport `size` to the Apollo store. "
  [{:keys [client]} size]
  (a/mutate! client {:mutation mutation :variables size}))

(defn- save-viewport!
  "Write the viewport `size` to the Apollo store and the cookies."
  [service size]
  (set-cookies! service size)
  (write-store! service size))

(defn- on-resize
  "Handle viewport resize changes."
  [service event]
  (let [size (size->map (.getSize (.-target event)))]
    (save-viewport! service size)
    (log/debug logger (viewport-change-msg size))))

(defrecord Viewport [client listener monitor]
  component/Lifecycle
  (start [service]
    (let [monitor (ViewportSizeMonitor.)
          listener (events/listen monitor EventType/RESIZE #(on-resize service %))]
      (some->> (viewport-size) (save-viewport! service))
      (assoc service :listener listener :monitor monitor)))

  (stop [service]
    (events/unlistenByKey listener)
    (assoc service :listener nil :monitor nil)))

(defn viewport
  "Returns a new viewport service component."
  []
  (-> (map->Viewport {})
      (component/using [:client])))
