(ns burningswell.web.ui.forms.signin
  (:require [apollo.core :as a]
            [burningswell.web.ui.buttons :as button]
            [burningswell.web.ui.social :as social]
            [burningswell.web.ui.text-field :as text-field]
            [burningswell.web.ui.links :as links]
            [clojure.string :as str]
            [sablono.core :refer [html]]))

(def default-data
  {:__typename "SigninForm"
   :state "ready"
   :id nil
   :login
   {:__typename "FormField"
    :errors []
    :value ""}
   :password
   {:__typename "FormField"
    :errors []
    :value ""}})

(def fsm
  {"ready"          {:missing-login    "login-error"
                     :missing-password "password-error"
                     :submit           "submitting"}
   "login-error"    {:change-login     "ready"}
   "password-error" {:change-password  "ready"}
   "submit-error"   {:change-login     "ready"
                     :change-password  "ready"}
   "submitting"     {:submit-success   "ready"
                     :submit-error     "submit-error"}})

(a/defgql signin-query
  '((query
     Signin
     [($id String)]
     (signin
      [(id $id)]
      ((client))
      id
      (login value errors)
      (password value errors)
      state
      __typename))))

(a/defgql signin-mutation
  '((mutation
     Signin
     [($input SigninInput!)]
     (signin
      [(input $input)]
      auth_token
      (user id username)))))

(defn- read-state [client form]
  (a/read-query client signin-query {:variables {:id (:id form)}}))

(defn- update-state! [client form f & args]
  (let [state (read-state client form)]
    (a/write-data! client (apply f state args))))

(defn- resolve-data [obj args context info]
  (if-let [id (:id args)]
    (assoc default-data :id id)
    default-data))

;; State machine

(defn- transition!
  "Updates app-state to contain the state reached by transitioning from the
  current state."
  [client form transition]
  (let [current-state (-> (read-state client form) :signin :state)
        next-state (get-in fsm [current-state transition])]
    ;; (prn "TRANSITION:" current-state "->" transition "->" next-state)
    (if next-state
      (update-state! client form assoc-in [:signin :state] next-state))))

(defn- on-login-change
  [client form value]
  (let [app-state (update-state! client form assoc-in [:signin :login :value] value)]
    (if (str/blank? value)
      (let [errors ["Login can't be blank."]]
        (update-state! client form assoc-in [:signin :login :errors] errors)
        (transition! client form :missing-login))
      (transition! client form :change-login))))

(defn- on-password-change
  [client form value]
  (let [app-state (update-state! client form assoc-in [:signin :password :value] value)]
    (if (str/blank? value)
      (let [errors ["Password can't be blank."]]
        (update-state! client form assoc-in [:signin :password :errors] errors)
        (transition! client form :missing-password))
      (transition! client form :change-password))))

(defn- assoc-errors [state field errors]
  (assoc-in state [:signin field :errors] errors))

(defn- assoc-value [state field value]
  (assoc-in state [:signin field :value] value))

(defn- reset-field [state field]
  (-> (assoc-errors state field [])
      (assoc-value field "")))

(defn- clear-form! [client form]
  (update-state! client form
                 #(-> (reset-field % :login)
                      (reset-field :password))))

(defn- on-submit-success
  [client form data]
  (clear-form! client form)
  (transition! client form :submit-success)
  (when-let [handler (:on-signin form)]
    (handler data)))

(defn- on-submit-error [client form data]
  (prn "FAILURE")
  (transition! client form :submit-error))

(defn- on-submit
  [client form action]
  (transition! client form :submit)
  (-> (action)
      (.then (fn [{:keys [data]}]
               (if-let [token (-> data :signin :auth_token)]
                 (on-submit-success client form data)
                 (on-submit-error client form data))))
      (.catch #(prn "TODO: Handle backend error"))))

(defn- field-value
  "Returns the value of `field-name` from the form `data`."
  [data field-name]
  (get-in data [:signin (keyword field-name) :value]))

(defn- form-state
  "Returns the state of the FSM from `data`."
  [data]
  (-> data :signin :state))

(defn- value
  "Returns the value of the `event` target."
  [event]
  (.. event -target -value))

(defn- login-error
  "Returns the login error message."
  [data]
  (case (form-state data)
    "login-error" (-> data :signin :login :errors first)
    nil))

(defn- login-field
  "Render a login text field."
  [client form data]
  (let [error (login-error data)]
    (text-field/login
     {:auto-complete "email"
      :helper-text
      {:persistent? (some? error)
       :text (or error "Enter your email or username.")
       :valid? (nil? error)
       :validation-message? (some? error)
       :validation? (some? error)}
      :on-change #(on-login-change client form (value %))
      :value (field-value data :login)})))

(defn- password-error
  "Returns the password error message."
  [data]
  (case (form-state data)
    "submit-error" "Invalid credentials."
    "password-error" (-> data :signin :password :errors first)
    nil))

(defn- password-field
  "Render a password field."
  [client form data]
  (let [error (password-error data)]
    (text-field/password
     {:helper-text
      {:persistent? (some? error)
       :text (or error "Enter your password.")
       :valid? (nil? error)
       :validation-message? (some? error)
       :validation? (some? error)}
      :on-change #(on-password-change client form (value %))
      :value (field-value data :password)})))

(defn- signin-variables
  "Returns the variables for the sign in mutation."
  [data]
  {:input {:login (field-value data :login)
           :password (field-value data :password)}})

(defn- submit-button
  "Render a submit button."
  [form data]
  (a/with-mutation [action {:keys [client]}]
    {:mutation signin-mutation
     :variables (signin-variables data)}
    (button/button
     "Signin"
     {:disabled (not= (form-state data) "ready")
      :raised true
      :on-click #(do (on-submit client form action)
                     (.preventDefault %))})))

(defn signin-form
  "Render a sign in form."
  [form]
  (a/with-query [{:keys [client data loading]}]
    {:query signin-query :variables {:id (:id form)}}
    (when-not loading
      (html [:form.signin-form
             [:div.signin-form__login
              (login-field client form data)]
             [:div.signin-form__password
              (password-field client form data)]
             [:div.signin-form__submit
              (submit-button form data)]
             [:div.signin-form__reset-password
              (links/reset-password)]
             [:div.signin-form__connect
              (social/connect)]
             [:div.signin-form__signup
              [:div "New to Burning Swell? "]
              (links/signup)]]))))
