(ns burningswell.web.router
  (:require [apollo.core :as a]
            [bidi.bidi :as bidi]
            [burningswell.web.history :as history]
            [burningswell.web.location :as location]
            [burningswell.web.logging :as log]
            [burningswell.web.session :as session]
            [cljs.loader :as loader]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [no.en.core :refer [format-url parse-url]]))

(def ^:private logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(def ^:private params
  "The regular expressions of path parameters."
  {:name [#".*" :name]
   :id [#"\d+" :id]})

(def ^:private slug-id-name
  "The route part of a slug."
  ["/" (:id params) "-" (:name params)])

(def ^:private ssr?
  (not (exists? js/window)))

(def routes
  "The Burning Swell web routes."
  ["/"
   {"" :home

    "continents"
    {"" :continents
     "/nearby" :continents-nearby
     slug-id-name :continent}

    "countries"
    {"" :countries
     "/nearby" :countries-nearby
     slug-id-name :country}

    "reset-password" :reset-password
    "map" :map

    "regions"
    {"" :regions
     "/nearby" :regions-nearby
     slug-id-name :region}

    "search" :search
    "settings" :settings
    "signin" :signin
    "signout" :signout
    "signup" :signup

    "spots"
    {"" :spots
     "/nearby" :spots-nearby
     slug-id-name :spot}

    "users"
    {"" :users
     "/nearby" :users-nearby
     slug-id-name :user}}])

(defn- current-browser-path []
  (when (exists? js/window)
    (.-pathname (.-location js/window))))

(defn- current-browser-url []
  (when (exists? js/window)
    (str (.-location js/window))))

(a/defgql route-query
  '((route
     ((client))
     handler
     loaded
     uri)))

(a/defgql create-page-view-mutation
  `((mutation
     create-page-view
     [($input CreatePageViewInput!)]
     (create-page-view
      [(input $input)]
      session-id url
      (location latitude longitude)
      (user id)))))

(defn- read-state [client]
  (a/read-query client route-query))

(defn- create-page-view!
  "Create a page view for the `route`."
  [{:keys [client location session]} route]
  (->> {:mutation create-page-view-mutation
        :variables
        {:input
         {:location (location/read-location location)
          :session_id (str (session/session-id session))
          :url (format-url route)}}}
       (a/mutate! client)))

(defn- write-route!
  "Write the `route` to the Apollo store."
  [{:keys [client] :as router} route loaded]
  (let [state (-> (read-state client)
                  (assoc-in [:route :handler] (:handler route))
                  (assoc-in [:route :uri] (:uri route))
                  (assoc-in [:route :loaded] loaded))]
    (a/write-data! client state)))

(defn- on-route-match
  [router route]
  (let [module (:handler route)]
    (log/info logger (str "Route matched: " (-> route :handler name)))
    (create-page-view! router route)
    (write-route! router route (if ssr? true (loader/loaded? module)))
    (when-not ssr? (loader/load module #(write-route! router route true)))
    route))

(defn- on-route-not-found [router request]
  (log/error logger (str "Can't match route: " (pr-str request))))

(defn- match-request [router request]
  (some->> (bidi/match-route (:routes router) (:uri request))
           (merge request)))

(defn- match-url [router url]
  (match-request router (parse-url url)))

(defn- safe-loaded? [module]
  (and (contains? loader/module-infos module)
       (loader/loaded? module)))

(defn browser-route
  "Returns the current browser route."
  [router]
  (when-let [route (match-url router (current-browser-url))]
    (assoc route :loaded (safe-loaded? (:handler route)))))

(defn match-request! [router request]
  (if-let [route (match-request router request)]
    (on-route-match router route)
    (on-route-not-found router request)))

(defn match-url! [router url]
  (match-request router (parse-url url)))

(defn match-path! [router path]
  (let [request (parse-url (current-browser-url))]
    (match-request! router request)))

(defn match-current-url!
  "Match the current browser url."
  [router]
  (match-path! router (current-browser-url)))

(defn- match-navigate-event! [router event]
  (match-path! router (str "/" (.-token event))))

(defn- register-listener!
  "Listen to HTML5 history navigation events."
  [router]
  (history/listen! (:history router) #(match-navigate-event! router %)))

(defrecord Router [history listener routes]
  component/Lifecycle
  (start [router]
    (let [listener (register-listener! router)]
      (log/info logger "Router started.")
      (assoc router :listener listener)))
  (stop [router]
    (history/unlisten! history listener)
    (log/info logger "Router stopped.")
    (assoc router :listener nil)))

(defrecord SSRRouter [routes request]
  component/Lifecycle
  (start [router]
    (match-request! router request)
    (log/info logger "SSR Router started.")
    router)
  (stop [router]
    (log/info logger "SSR Router stopped.")
    router))

(defn router
  "Returns a new router component."
  [& [config]]
  (-> (assoc config :routes routes)
      (map->Router)
      (component/using [:client :history :logging :location :session])))

(defn ssr-router
  "Returns a new server side rendering router component."
  [request & [config]]
  (-> (assoc config :routes routes :request request)
      (map->SSRRouter)
      (component/using [:client :logging :location :session])))

(defn path-for [router handler & [params]]
  (apply bidi/path-for (:routes router) handler (mapcat concat params)))

(defn to! [router handler & [params]]
  (if-let [path (path-for router handler params)]
    (history/set-path! (:history router) path)
    (throw (ex-info (str "Unknown handler: " (name handler))
                    {:handler handler :params params}))))

(defn route-consumer [render-fn & [opts]]
  (a/query (assoc opts :query route-query)
           (fn [{:keys [data]}]
             (render-fn (a/get data :route)))))
