(ns burningswell.worker.addresses
  (:require [burningswell.db.addresses :as addresses]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.rabbitmq.core :as rabbitmq]
            [burningswell.worker.util :as util]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [geocoder.google :as geocoder]
            [langohr.basic :as basic]
            [langohr.exchange :as exchange]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]))

(s/defrecord Worker
    [channel :- (s/maybe Channel)
     subscriptions :- [s/Any]]
  {s/Any s/Any})

(defn country
  "Return the country for `location` and `result`."
  [db location result]
  (when-let [country (geocoder/country result)]
    (or (countries/closest db location)
        (countries/by-iso-3166-1-alpha-2 db (:iso-3166-1-alpha-2 country)))))

(defn region
  "Return the region for `location` and `result`."
  [db location result]
  (or (regions/closest db location)
      (regions/by-name db (geocoder/region result))))

(defn address-by-location
  "Return the address for `location`."
  [db location]
  (when-let [result (first (geocoder/geocode-location location))]
    {:_embedded
     {:country (country db location result)
      :region (region db location result)}
     :city (geocoder/city result)
     :formatted (:formatted-address result)
     :location location
     :postal-code (geocoder/postal-code result)
     :street-name (geocoder/street-name result)
     :street-number (geocoder/street-number result)}))

(defn geocode-address
  "Geocode and save the address of `location`."
  [db location]
  (when-let [address (address-by-location db location)]
    (if-let [found (addresses/by-formatted db (:formatted address))]
      (addresses/update db (merge found address))
      (addresses/insert db address))))

(s/defn ^:always-validate on-spot-created
  "Geocode the address of spots that have been created."
  [worker :- Worker channel :- Channel metadata :- s/Any spot :- Spot]
  (if-let [address (geocode-address (:db worker) (:location spot))]
    (log/infof "Saved address %s (location %s, id %s)."
               (:formatted address) (:id address) (:location address))
    (log/warnf "Can't geocode address for location %s."
               (:location spot))))

(s/defn ^:always-validate on-spot-updated
  "Geocode the address of spots that have been updated."
  [worker :- Worker channel :- Channel metadata :- s/Any spot :- Spot]
  (on-spot-created worker channel metadata spot))

(extend-protocol component/Lifecycle
  Worker
  (start [worker]
    (util/start-subscriber worker))
  (stop [worker]
    (util/stop-subscriber worker)))

(def subscriptions
  [{:name ::spot-created
    :exchange {:name "api" :type :topic :durable true}
    :handler on-spot-created
    :routing-keys ["spots.created"]
    :queue {:name "addresses.spots.created"
            :auto-delete false
            :durable true}
    :dead-letter true}

   {:name ::spot-updated
    :exchange {:name "api" :type :topic :durable true}
    :handler on-spot-updated
    :routing-keys ["spots.updated"]
    :queue {:name "addresses.spots.updated"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new address worker."
  [& [config]]
  (component/using
   (map->Worker (assoc config :subscriptions subscriptions))
   [:broker :db]))
