(ns burningswell.worker.subscriber
  (:require [burningswell.rabbitmq.core :as rabbitmq]
            [com.stuartsierra.component :as component]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.exchange :as exchange]
            [langohr.queue :as queue]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]))

(s/defrecord Subscriber
    [channel :- (s/maybe Channel)
     subscriptions :- [s/Any]]
  {s/Any s/Any})

(defn assoc-channel [worker]
  (let [channel (rabbitmq/open-channel (:broker worker))]
    (assoc worker :channel channel)))

(defn- bind-queue [worker queue exchange routing-keys & [opts]]
  (doseq [routing-key routing-keys]
    (queue/bind (:channel worker)
                (name (:name queue))
                (name (:name exchange))
                (merge opts {:routing-key (name routing-key)}))))

(defn dead-letter
  "Return the dead letter exchange or queue for `s`."
  [{:keys [name] :as m}]
  (when m (update-in m [:name] #(str % ".dead"))))

(defn- declare-exchange [worker subscription]
  (when-let [exchange (:exchange subscription)]
    (exchange/declare
     (:channel worker)
     (name (:name exchange))
     (name (or (:type exchange) :direct))
     (dissoc exchange :name))
    exchange))

(defn queue-options [subscription]
  (let [{:keys [exchange queue]} subscription
        opts (dissoc queue :name)]
    (if (:dead-letter subscription)
      (update-in
       opts [:arguments ] merge
       {"x-dead-letter-exchange"
        (name (:name (dead-letter exchange)))
        "x-dead-letter-routing-key"
        (-> (dead-letter queue) :name name)})
      opts)))

(defn- declare-queue
  [worker subscription]
  (let [{:keys [exchange queue]} subscription]
    (queue/declare
     (:channel worker)
     (name (:name queue))
     (queue-options subscription))
    queue))

(defn- declare-dead-letter-exchange
  "Declare a dead letter exchange for the subscription."
  [worker subscription]
  (when (:dead-letter subscription)
    (let [exchange (:exchange subscription)
          dead-exchange (dead-letter exchange)]
      (exchange/declare
       (:channel worker)
       (name (:name dead-exchange))
       "direct"
       (dissoc exchange :name))
      dead-exchange)))

(defn- declare-dead-letter-queue
  "Declare a dead letter queue for the subscription."
  [worker subscription]
  (when (:dead-letter subscription)
    (let [{:keys [exchange queue]} subscription
          dead-queue (dead-letter queue)
          opts (dissoc queue :name)]
      (queue/declare
       (:channel worker)
       (name (:name dead-queue))
       opts)
      dead-queue)))

(defn wrap-exceptions
  "Return a wrapped `handler` that acks a messages if `handler` didn't
  raise an exception, otherwise rejects the message. "
  [handler]
  (fn [channel {:keys [delivery-tag] :as metadata} payload]
    (try (handler channel metadata payload)
         (if (channel/open? channel)
           (basic/ack channel delivery-tag)
           (log/warn {:msg "Channel closed, can't ack message."
                      :metadata metadata
                      :payload payload}))
         (catch Exception e
           (log/error e "Exception caught. Rejecting message.")
           (if (channel/open? channel)
             (basic/reject channel delivery-tag false)
             (log/warn {:msg "Channel closed, can't reject message."
                        :metadata metadata
                        :payload payload}))))))

(defn subscribe [worker]
  (let [channel (:channel worker)]
    (reduce
     (fn [worker {:keys [exchange handler routing-keys queue]
                  :as subscription}]
       (let [exchange (declare-exchange worker subscription)
             dead-exchange (declare-dead-letter-exchange worker subscription)
             queue (declare-queue worker subscription)
             dead-queue (declare-dead-letter-queue worker subscription)]
         (bind-queue worker queue exchange routing-keys)
         (when (and dead-exchange dead-queue)
           (queue/bind
            (:channel worker)
            (-> dead-queue :name name)
            (-> dead-exchange :name name)
            {:routing-key (-> dead-queue :name name)}))
         (update-in worker [:subscriptions] conj
                    (assoc subscription :tag
                           (rabbitmq/subscribe
                            (:channel worker)
                            (:name queue)
                            (wrap-exceptions (partial handler worker)))))))
     (assoc worker :subscriptions [])
     (:subscriptions worker))))

(defn unsubscribe [worker]
  (let [channel (:channel worker)]
    (reduce
     (fn [worker {:keys [tag] :as subscription}]
       (rabbitmq/unsubscribe (:channel worker) tag)
       (update-in worker [:subscriptions] conj
                  (dissoc subscription :tag)))
     (assoc worker :subscriptions [])
     (:subscriptions worker))))

(s/defn ^:always-validate start-subscriber :- Subscriber
  "Start the RabbitMQ subscriber."
  [subscriber :- Subscriber]
  (let [subscriber (assoc-channel subscriber)
        subscriber (subscribe subscriber)]
    (log/info "RabbitMQ subscriber started.")
    subscriber))

(s/defn ^:always-validate stop-subscriber :- Subscriber
  "Stop the RabbitMQ subscriber."
  [subscriber :- Subscriber]
  (when-let [channel (:channel subscriber)]
    (unsubscribe subscriber)
    (rabbitmq/close-channel channel)
    (log/info "RabbitMQ subscriber stopped."))
  (assoc subscriber :channel nil))

(extend-protocol component/Lifecycle
  Subscriber
  (start [worker]
    (start-subscriber worker))
  (stop [worker]
    (stop-subscriber worker)))

(defn subscriber
  "Return a new subscriber."
  [subscriptions & [config]]
  (-> (assoc config :subscriptions subscriptions)
      (map->Subscriber)
      (component/using [:broker :db])))
