(ns burningswell.worker.photos
  (:require [aws.sdk.s3 :as s3]
            [burningswell.db.images :as images]
            [burningswell.db.photos :as photos]
            [burningswell.db.schemas :refer :all]
            [burningswell.worker.subscriber :refer [subscriber]]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [image-resizer.core :as resizer]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]]
            [schema.core :as s]
            [slingshot.slingshot :refer [try+]])
  (:import burningswell.worker.subscriber.Subscriber
           com.rabbitmq.client.Channel
           java.awt.image.BufferedImage
           [java.io ByteArrayInputStream ByteArrayOutputStream]
           javax.imageio.ImageIO))

(s/defschema Dimension
  {:label s/Str
   (s/optional-key :width) s/Int
   (s/optional-key :height) s/Int})

(defn- bytes->buffered-image
  "Convert `bytes` into a buffered image."
  [bytes]
  (ImageIO/read (ByteArrayInputStream. bytes)))

(defn- buffered-image->bytes
  "Convert `buffered-image` into a byte in `format`."
  [^BufferedImage buffered-image ^String format]
  (let [baos (ByteArrayOutputStream.)]
    (assert (ImageIO/write buffered-image format baos)
            (str "Can't write image in format: " format))
    (.toByteArray baos)))

(defn- content-md5
  "Return the MD5 sum of `bytes`."
  [bytes]
  (base64-encode (digest/md5-bytes bytes)))

(defn- content-type
  "Return the content type of `response`."
  [response]
  (get-in response [:headers "content-type"]))

(defn content-type-extension
  "Returns the filename extension from `content-type`."
  [content-type]
  (if-not (str/blank? content-type)
    (last (str/split content-type #"/"))))

(defn- content-disposition
  "Return the content disposition header of `photo`."
  [photo label content-type]
  (str "attachment; filename=photo-"
       (:id photo) "-" label "."
       (content-type-extension content-type)))

(s/defn ^:always-validate s3-image-key :- s/Str
  "Return the S3 key for `photo`, `label` and `content-type`."
  [worker :- Subscriber photo :- Photo label :- s/Str content-type :- s/Str]
  (let [extension (content-type-extension content-type)]
    (str (-> worker :s3 :photos :prefix)
         "/" (:id photo)
         "/" label
         "." extension)))

(s/defn ^:always-validate s3-image-url :- s/Str
  "Return the S3 url for `photo`, `label` and `content-type`."
  [worker :- Subscriber photo :- Photo label :- s/Str content-type :- s/Str]
  (format "https://%s.s3.amazonaws.com/%s"
          (-> worker :s3 :photos :bucket)
          (s3-image-key worker photo label content-type)))

(s/defn ^:always-validate download-image
  "Download the original image of `photo`."
  [worker :- Subscriber photo :- Photo]
  (let [response (http/get (:url photo) {:as :byte-array})]
    (log/infof "Downloaded photo %s." (:id photo))
    (if-let [image (bytes->buffered-image (:body response))]
      {:buffered image
       :bytes (:body response)
       :content-type (content-type response)
       :label "original"}
      (throw (ex-info "Can't decode photo."
                      {:type :decode-error
                       :photo photo
                       :response response})))))

(s/defn ^:always-validate save-image-to-s3 :- Image
  "Upload the `image`."
  [worker :- Subscriber image :- Image]
  (s3/put-object
   (:aws worker) (-> worker :s3 :photos :bucket) (:s3-key image)
   (ByteArrayInputStream. (:bytes image))
   {:cache-control "public, max-age 31536000"
    :content-disposition (:content-disposition image)
    :content-length (:content-length image)
    :content-md5 (:content-md5 image)
    :content-type (:content-type image)
    :expires "2034-01-01T00:00:00Z"}
   (s3/grant :all-users :read))
  (log/infof "Saved image %s to S3 %s." (:id image) (:url image))
  image)

(s/defn ^:always-validate save-image-to-db :- Image
  "Save the `image` to the database."
  [worker :- Subscriber photo :- Photo image]
  (let [{:keys [buffered content-type label]} image]
    (->> {:_embedded {:photo photo}
          :content-disposition (content-disposition photo label content-type)
          :content-length (count (:bytes image))
          :content-md5 (content-md5 (:bytes image))
          :height (.getHeight buffered)
          :s3-key (s3-image-key worker photo label content-type)
          :url (s3-image-url worker photo label content-type)
          :width (.getWidth buffered)}
         (merge image)
         (images/save (:db worker))
         (merge image))))

(s/defn ^:always-validate resize-image
  "Resize `image` to `dimension`."
  [image :- Image dimension :- Dimension]
  (let [{:keys [label width height]} dimension
        buffered (:buffered image)
        resized (cond
                  (and width height)
                  (resizer/resize buffered width height)
                  width
                  (resizer/resize-to-width buffered width)
                  height
                  (resizer/resize-to-height buffered height))]
    {:buffered resized
     :bytes (buffered-image->bytes resized "png")
     :content-type "image/png"
     :label label}))

(s/defn ^:always-validate import-image :- Image
  "Import the `image` of `photo`."
  [worker :- Subscriber photo :- Photo image]
  (->> (save-image-to-db worker photo image)
       (save-image-to-s3 worker)))

(s/defn ^:always-validate download-and-import :- Image
  "Download the original image of `photo` and import it."
  [worker :- Subscriber photo :- Photo]
  (->> (download-image worker photo)
       (import-image worker photo)))

(s/defn ^:always-validate resize-and-import :- Image
  "Resize the `original` image of `photo` to `dimension` and import it."
  [worker :- Subscriber photo :- Photo original :- Image dimension]
  (->> (resize-image original dimension)
       (import-image worker photo)))

(s/defn ^:always-validate update-photo-status :- Photo
  "Update the `status` of `photo` in the database."
  [worker :- Subscriber photo :- Photo status :- s/Keyword]
  (photos/update (:db worker) (assoc photo :status (name status))))

(s/defn ^:always-validate import-photo :- s/Any
  "Import the `photo`."
  [worker :- Subscriber photo :- Photo]
  (try+
   (let [original (download-and-import worker photo)]
     (doseq [dimension (:dimensions worker)]
       (resize-and-import worker photo original dimension))
     (log/infof "Photo %s successfully imported." (:id photo))
     (update-photo-status worker photo :finished))
   (catch [:type :decode-error] _
     (log/errorf "Photo import error: Can't decode image." (:id photo))
     (update-photo-status worker photo :decode-error))
   (catch [:status 401] _
     (log/errorf "Photo import error: Image download forbidden." (:id photo))
     (update-photo-status worker photo :forbidden))
   (catch [:status 404] _
     (log/errorf "Photo import error: Image not found." (:id photo))
     (update-photo-status worker photo :not-found))))

(s/defn ^:always-validate on-photo-created
  "Handle created photos."
  [worker :- Subscriber channel :- Channel metadata :- s/Any photo :- Photo]
  (import-photo worker photo))

(s/defn ^:always-validate on-photo-deleted
  "Handle deleted photos."
  [worker :- Subscriber channel :- Channel metadata :- s/Any photo :- Photo]
  ;; TODO: Delete photo
  nil)

(s/defn ^:always-validate on-photo-updated
  "Handle updated photos."
  [worker :- Subscriber channel :- Channel metadata :- s/Any photo :- Photo]
  (import-photo worker photo))

(def dimensions
  "The dimension used by the photos worker to scale images."
  [{:label "tiny" :width 320}
   {:label "small" :width 480}
   {:label "medium" :width 640}
   {:label "large" :width 800}
   {:label "huge" :width 1024}])

(def subscriptions
  "The RabbitMQ subscriptions of the photos worker."
  [{:name ::photo-created
    :exchange {:name "api" :type :topic :durable true}
    :handler on-photo-created
    :routing-keys ["photos.created"]
    :queue {:name "photos.created"
            :auto-delete false
            :durable true}
    :dead-letter true}
   {:name ::photo-deleted
    :exchange {:name "api" :type :topic :durable true}
    :handler on-photo-deleted
    :routing-keys ["photos.deleted"]
    :queue {:name "photos.deleted"
            :auto-delete false
            :durable true}
    :dead-letter true}
   {:name ::photo-updated
    :exchange {:name "api" :type :topic :durable true}
    :handler on-photo-updated
    :routing-keys ["photos.updated"]
    :queue {:name "photos.updated"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new photos worker."
  [& [config]]
  (->> (assoc config :dimensions dimensions)
       (subscriber subscriptions)))
