(ns burningswell.worker.import
  (:require [burningswell.file :refer [edn-line-seq]]
            [burningswell.config.core :as config]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.rabbitmq.client :as rabbitmq]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [commandline.core :refer [print-help with-commandline]]
            [environ.core :refer [env]]
            [kithara.core :as k]))

(defn full-name [spot region country]
  (->> [(:name spot) (:name region) (:name country)]
       (remove str/blank?)
       (str/join ", ")))

(defn save-spot
  "Save the `spot` to `db`."
  [db spot]
  (when-let [location (:location spot)]
    (let [country (countries/closest db location)
          region (regions/closest db location)
          spot-name (full-name spot region country)]
      (if (empty? (spots/within-distance db location 0.3))
        (let [spot (spots/insert db {:_embedded
                                     {:country country
                                      :region region}
                                     :name (:name spot)
                                     :location location})]
          (log/infof "%s imported." spot-name)
          spot)
        (log/infof "%s not imported, it already exists." spot-name)))))

(defmulti process-message
  "Handle import messages."
  (fn [message] (-> message :routing-key keyword)))

(defmethod process-message :spot
  [{:keys [body channel env] :as msg}]
  (when-let [spot (save-spot (:db env) body)]
    (rabbitmq/publish-edn
     channel {:body spot
              :exchange "api"
              :routing-key "spots.created"})
    (log/info {:msg "Spot imported" :spot spot}))
  {:status :ack})

(defn worker
  "Return a new address worker."
  [config]
  (-> process-message
      (k/consumer {:as rabbitmq/read-edn :consumer-name "Spot Importer"})
      (k/with-channel {:prefetch-count 1})
      (k/with-durable-queue "worker.import"
        {:exchange "import" :routing-keys ["spot"]})
      (k/with-connection (rabbitmq/config (:broker config)))
      (k/with-env config)
      (component/using [:db :topology])))

(defn import-spots
  "Import the spots from `input`."
  [broker input]
  (rabbitmq/with-connection [broker broker]
    (with-open [reader (io/reader input)]
      (doseq [spot (edn-line-seq reader)]
        (rabbitmq/publish-edn
         broker {:exchange "import"
                 :body spot
                 :routing-key "spot"})))))

(defn -main [& args]
  (with-commandline [[opts input] args]
    [[h help "Print this help."]]
    (when (or (:help opts) (str/blank? input))
      (print-help "import FILE")
      (System/exit 1))
    (import-spots (config/broker env) input)
    (System/exit 0)))
