(ns burningswell.worker.vision
  (:require [burningswell.io :as io]
            [burningswell.time :refer [current-duration-str]]
            [clj-time.core :as time]
            [clojure.spec :as s]
            [clojure.spec.gen :as gen]
            [clojure.string :as str]
            [clojure.tools.logging :as log])
  (:import com.google.api.client.googleapis.auth.oauth2.GoogleCredential
           com.google.api.client.googleapis.javanet.GoogleNetHttpTransport
           com.google.api.client.json.jackson2.JacksonFactory
           [com.google.api.services.vision.v1
            Vision Vision$Builder VisionScopes]
           [com.google.api.services.vision.v1.model
            AnnotateImageRequest AnnotateImageResponse
            BatchAnnotateImagesRequest Feature Image]
           com.google.common.collect.ImmutableList
           [java.nio.file Files Paths]))

(s/def ::description string?)

(s/def ::mid
  (s/with-gen string?
    #(gen/fmap (fn [id] (str "/m/" id)) (gen/not-empty (s/gen string?)))))

(s/def ::score
  (s/double-in :min 0.0 :max 1.0 :NaN? false :infinite? false))

(defn service?
  "Returns true if `x` is a Vision service, otherwise false."
  [x]
  (instance? Vision x))

(defn response?
  "Returns true if `x` is an annotated image response, otherwise false."
  [x]
  (instance? AnnotateImageResponse x))

(defn service
  "Return the Google Cloud Vision service."
  []
  (let [credentials (GoogleCredential/getApplicationDefault)]
    (log/infof "Google Vision service initialized using service account %s."
               (or (.getServiceAccountId credentials) "n/a"))
    (-> (Vision$Builder.
         (GoogleNetHttpTransport/newTrustedTransport)
         (JacksonFactory/getDefaultInstance)
         (.createScoped credentials (VisionScopes/all)))
        (.setApplicationName "burningswell")
        (.build))))

(defn feature
  "Return a `Feature` instance."
  [type & [max-results]]
  (-> (Feature.)
      (.setType
       (case type
         :faces "FACE_DETECTION"
         :labels "LABEL_DETECTION"
         :landmarks "LANDMARK_DETECTION"))
      (.setMaxResults (int (or max-results 5)))))

(s/fdef feature
  :args (s/cat :type #{:faces :labels :landmarks}
               :max-results (s/? pos-int?)))

(defn slurp-image [image]
  (if (bytes? image)
    image (io/slurp-byte-array image)))

(defn- annotate-request
  "Return the annotate request for `image`."
  [image & [{:keys [faces labels landmarks]}]]
  (-> (AnnotateImageRequest.)
      (.setImage (.encodeContent (Image.) (slurp-image image)))
      (.setFeatures
       (ImmutableList/of
        (->> [(when labels (feature :labels labels))
              (when landmarks (feature :landmarks landmarks))
              (when faces (feature :faces faces))]
             (remove nil?))))))

(defn- annotate-batch-request
  "Return the batch annotate request for `images`."
  [images & [opts]]
  (.setRequests
   (BatchAnnotateImagesRequest.)
   (map #(annotate-request %1 opts) images)))

(defn- extract-annotation
  "Convert an entity annotation into a Clojure map."
  [annotation]
  (->> (for [[k v] (into {} annotation)]
         [(keyword k) v])
       (into {})))

(defn- extract-annotations
  "Convert an entity annotation into a Clojure map."
  [response]
  (->> (for [[type annotation]
             [[:faces (.getFaceAnnotations response)]
              [:labels (.getLabelAnnotations response)]
              [:landmarks (.getLandmarkAnnotations response)]]
             :when (not (empty? annotation))]
         {type (mapv extract-annotation annotation)})
       (apply merge)))

(defn annotate-images
  "Annotate `images` using the Google Cloud Vision `service`."
  [service images & [opts]]
  (let [started-at (time/now)
        requests (annotate-batch-request images opts)
        annotate (.annotate (.images service) requests)]
    (.setDisableGZipContent annotate true)
    (let [responses (.. annotate execute getResponses)]
      (log/infof "Annotated batch of %s images in %s." (count images)
                 (current-duration-str started-at))
      (mapv extract-annotations responses))))

(s/fdef annotate-images
  :args (s/cat :service service? :images any? :opts (s/? (s/nilable map?))))

(defn annotate-image
  "Annotate `images` using the Google Cloud Vision `service`."
  [service image & [opts]]
  (first (annotate-images service [image] opts)))

(s/fdef annotate-image
  :args (s/cat :service service? :images any? :opts (s/? (s/nilable map?))))
