(ns burningswell.worker.flickr
  (:require [burningswell.db.photos :as photos]
            [burningswell.db.spots :as spots]
            [burningswell.db.photo-labels :as photo-labels]
            [burningswell.flickr :as flickr]
            [burningswell.rabbitmq.client :as rabbitmq]
            [burningswell.worker.vision :as vision]
            [clj-http.client :as http]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [geo.core :as geo]
            [geo.postgis :refer [point]]
            [kithara.core :as k]
            [kithara.patterns.dead-letter-backoff
             :refer [with-durable-dead-letter-backoff]]
            [medley.core :refer [distinct-by]]
            [sqlingvo.core :as sql]
            [metrics.timers :refer [deftimer time!]]
            [clojure.edn :as edn]
            [burningswell.db.countries :as countries]))

(def queue
  "The Flickr worker queue."
  {:auto-delete? false
   :durable? true
   :exclusive? false
   :name "worker.flickr"})

(deftimer search-photos-by-criteria-timer)
(deftimer download-image-timer)
(deftimer label-photos-timer)

(def commercial-licenses
  "Flickr licenses that allow photos to be used on commercial sites."
  (map :id (filter :usable? flickr/licenses)))

(def include-labels
  "Photos with these labels are included."
  #{"coast" "ocean" "surf" "surfing" "wave"})

(def exclude-labels
  "Photos with these labels are excluded."
  #{"man" "woman" "vacation"})

(defn search-criterias
  "Return a vector of search criterias for `spot`."
  [country {:keys [location] :as spot}]
  (let [license (str/join ","  commercial-licenses)]
    [{:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str (:name spot) ", " (:name country))
      :radius 0.5
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 0.5
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text "surf"
      :radius 0.5
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :tags "surf"
      :radius 0.5
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str (:name spot) ", " (:name country))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :tags "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :text (str (:name spot) ", " (:name country))
      :tags "surf"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "surf"
      :sort "relevance"}]))

(defn parse-location
  "Parse the Flickr location response."
  [{:keys [latitude longitude]}]
  (when (and latitude longitude)
    (point 4326 (Double/parseDouble longitude) (Double/parseDouble latitude))))

(defn parse-size
  "Parse the Flickr photo size response. The Flickr API returns width
  and height sometimes a string, sometimes as an integer."
  [{:keys [width height] :as size}]
  (cond-> size
    (string? width)
    (update :width #(Long/parseLong %))
    (string? height)
    (update :height #(Long/parseLong %))))

(defn label-name
  "Return the lower cased and trimmed label for `s`."
  [s]
  (some-> s name str/lower-case str/trim))

(defn label-size
  "Return the size of `photo` used for labeling."
  [sizes min-width max-width]
  (let [sorted (sort-by :width sizes)]
    (or (last (filter #(and (>= (:width %) min-width)
                            (<= (:width %) max-width))
                      sorted))
        (first (filter #(> (:width %) max-width) sorted))
        (last (filter #(< (:width %) min-width) sorted)))))

(defn log-labels
  "Log the `labels` from the Vision API."
  [labels]
  (log/infof "  Vision API Labels:  %s"
             (str/join ", "(map :description labels))))

(defn log-search-criteria
  "Log the Flickr search criteria."
  [{:keys [lat lon text radius tags sort]} photos]
  (log/infof "Found %s Flickr photos for search criteria:" (count photos))
  (when (and lat lon) (log/infof "  Location: %s, %s" lat lon))
  (when text (log/infof "  Text:     %s" text))
  (when radius (log/infof "  Radius:   %s km" radius))
  (when tags (log/infof "  Tags:     %s" tags))
  (when sort (log/infof "  Sort:     %s" sort)))

(defn save-photo-label-scores
  "Save the scores `labels` of the Flickr `photo`."
  [db photo labels]
  (when-not (empty? labels)
    (let [values (map #(vector (:id photo) (:mid %) (:score %)) labels)]
      @(sql/insert db :photo-label-scores [:photo-label-id :photo-id :score]
         (sql/select db [:photo-labels.id :v.photo-id :v.score]
           (sql/from (sql/as (sql/values db values)
                             :v [:photo-id :mid :score]))
           (sql/join :photo-labels '(on (= :photo-labels.mid :v.mid))))
         (sql/on-conflict [:photo-label-id :photo-id]
           (sql/do-update {:score :EXCLUDED.score}))))))

(defn save-photo-labels
  "Save the labels of the Flickr `photo` to `db`."
  [db labels]
  (photo-labels/save-all! db labels))

(defn save-photo
  "Save the Flickr `photo` of `spot`."
  [db spot photo]
  (let [saved (photos/save! db photo)]
    (spots/add-photo! db spot saved)
    (let [labels (:labels photo)]
      (save-photo-labels db labels)
      (save-photo-label-scores db saved labels)
      (log/infof "Saved photo #%d \"%s\" for spot #%d \"%s\"."
                 (:id saved) (:title saved)
                 (:id spot) (:name spot))
      (log/infof "  Flickr source URL:  %s" (:url saved))
      (log-labels labels))
    (assoc photo :id (:id saved))))

(defn publish-photo
  "Publish `photo` for processing."
  [channel photo]
  (rabbitmq/publish-edn
   channel {:body photo
            :exchange "api"
            :routing-key "photos.created"})
  (log/infof "Published photo #%s \"%s\"." (:id photo) (:title photo))
  photo)

(defn publish-photos
  "Publish `photos` for processing."
  [channel photos]
  (doseq [photo photos] (publish-photo channel photo)))

(defn exclude-existing-photos
  "Return a transducer that removes already saved photos."
  [db]
  (remove #(photos/by-flickr-id db (:id %))))

(defn size-by-name
  "Find the `size` of `name`."
  [sizes name]
  (let [name (str/lower-case name)]
    (first (filter #(= (str/lower-case (:label %)) name) sizes))))

(defn search-photos-by-criteria
  "Search photos of `spot` on Flickr by `criteria`."
  [db flickr spot criteria]
  (time! search-photos-by-criteria-timer
    (let [photos (flickr/search flickr criteria)]
      (log-search-criteria criteria photos)
      (->> (into [] (exclude-existing-photos db) photos)
           (pmap (fn [photo]
                   (let [sizes (flickr/sizes flickr (:id photo))]
                     {:label-size (label-size (mapv parse-size sizes) 300 800)
                      :spot (select-keys spot [:id :name])
                      :flickr-id (Long/parseLong (:id photo))
                      :flickr-owner-id (:owner photo)
                      :url (:source (size-by-name sizes "original"))
                      :title (:title photo)})))))))

(defn download-image
  "Download the image used for labeling."
  [photo]
  (if-let [url (-> photo :label-size :source)]
    (time! download-image-timer
      (try (->> {:as :byte-array
                 :throw-exceptions true}
                (http/get url)
                :body
                (assoc photo :bytes))
           (catch Exception e
             (log/errorf "Can't download image. HTTP status %s."
                         (-> e ex-data :status)))))
    (log/error "Can't download image. No appropriate size found.")))

(defn label-photos
  "Label the `photos` using the Google Vision API."
  [vision-service spot photos]
  (time! label-photos-timer
    (mapv (fn [photo {:keys [labels] :as response}]
            (log/infof "Labeled photo \"%s\" for spot #%d \"%s\"."
                       (:title photo) (:id spot) (:name spot))
            (log/infof "  Flickr source URL:  %s" (:url photo))
            (log-labels labels)
            (assoc photo :labels (vec labels)))
          photos
          (vision/annotate-images
           vision-service (map :bytes photos)
           {:labels 10}))))

(defn label-photos-in-batches
  "Returns a transducer that downloads and labels photos."
  [vision spot]
  (comp (partition-all 10)
        (mapcat #(label-photos vision spot %))
        (map #(dissoc % :bytes))))

(defn update-photo-details
  "Fetch the details of the Flickr `photo` and return an updated photo."
  [flickr photo]
  (let [details (flickr/photo-info flickr (:id photo))]
    (assoc photo :location (parse-location (:location details)))))

(defn update-photo-owner
  "Fetch the owner of the Flickr `photo` and return an updated photo."
  [flickr photo]
  (let [user (flickr/user-info flickr (:flickr-owner-id photo))]
    (-> (assoc photo :flickr-owner-name (-> user :username :_content))
        (assoc :flickr-owner-url (-> user :profileurl :_content)))))

(defn photo-label-set
  "Return the labels of `photo` as a set."
  [photo]
  (set (map (comp label-name :description)
            (-> photo :labels))))

(defn include-photos-labeled
  "Include photos that are labeled with any of `labels`."
  [labels]
  (let [labels (set (map label-name labels))]
    (remove #(empty? (set/intersection (photo-label-set %1) labels)))))

(defn exclude-photos-labeled
  "Exclude photos that are labeled with any of `labels`."
  [labels]
  (let [labels (set (map label-name labels))]
    (filter #(empty? (set/intersection (photo-label-set %1) labels)))))

(defn make-pipeline
  [db flickr vision spot max-photos]
  (comp (mapcat #(search-photos-by-criteria db flickr spot %1))
        (distinct-by :flickr-id)
        (exclude-existing-photos db)
        (map download-image)
        (remove nil?)
        (label-photos-in-batches vision spot)
        (include-photos-labeled include-labels)
        (exclude-photos-labeled exclude-labels)
        (map (partial update-photo-details flickr))
        (map (partial update-photo-owner flickr))
        (map (partial save-photo db spot))
        (take max-photos)))

(defn import-photos
  "Import Flickr photos for `spot`."
  [db flickr spot]
  (when-let [spot (spots/by-id db (:id spot))]
    (with-open [vision (vision/service)]
      (let [country (countries/by-spot db spot)
            flickr (flickr/client (:api-key flickr))
            search-criterias (search-criterias country spot)
            pipeline (make-pipeline db flickr vision spot 3)
            photos (into [] pipeline search-criterias)]
        (when-not (empty? photos)
          (spots/save-cover-photo! db spot (first photos)))
        photos))))

(defmulti process-message
  "Handle messages that require geocoding of addresses."
  (fn [message] (-> message :routing-key keyword)))

(defmethod process-message :spots.created
  [{:keys [body channel env]}]
  (->> (import-photos (:db env) (:flickr env) body)
       (publish-photos channel))
  {:status :ack})

(defmethod process-message :spots.updated
  [{:keys [body channel env]}]
  (when (empty? (photos/by-spot (:db env) body))
    (->> (import-photos (:db env) (:flickr env) body)
         (publish-photos channel)))
  {:status :ack})

(defn worker
  "Return a new Flickr worker."
  [broker config]
  (-> process-message
      (k/consumer {:as rabbitmq/read-edn
                   :consumer-name "Flickr Photo Importer"})
      (with-durable-dead-letter-backoff)
      (k/with-durable-queue (:name queue) queue
        {:exchange "api" :routing-keys ["spots.created" "spots.updated"]})
      (k/with-channel {:prefetch-count 1})
      (k/with-connection (rabbitmq/config broker))
      (k/with-env config)
      (component/using [:db :topology])))
