(ns burningswell.worker.weather.models
  (:require [burningswell.db.connection :as db]
            [burningswell.db.weather.models :as models]
            [burningswell.rabbitmq.client :as rabbitmq]
            [burningswell.worker.config :refer [config]]
            [burningswell.worker.topology :as topology]
            [clojure.string :as str]
            [taoensso.timbre :as log]
            [com.stuartsierra.component :as component]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [kithara.core :as k]
            [kithara.patterns.dead-letter-backoff :as dead-letter]
            [netcdf.dods :as dods]
            [clojure.spec.alpha :as s]))

(defn- latest-reference-time
  [datasources]
  (last (sort (map :reference-time datasources))))

(defn update-latest-reference-time!
  "Update the latest reference time of `model`."
  [db {:keys [datasources] :as model}]
  (let [reference-time (latest-reference-time datasources)]
    (->> (first @(sql/update db :weather.models
                   {:latest-reference-time reference-time}
                   (sql/where `(= :id ~(:id model)))
                   (sql/returning :*)))
         (merge model))))

(defn- datasource->row
  "Return a data source row for `model` and `datasource`."
  [model datasource]
  (-> (select-keys datasource [:das :dds :dods :reference-time])
      (assoc :model-id (:id model))))

(defn insert-datasources!
  "Update the data sources of `model`."
  [db  {:keys [datasources] :as model}]
  (->> @(sql/insert db :weather.datasources []
          (sql/values (map #(datasource->row model %) datasources))
          (sql/on-conflict [:model-id :reference-time]
            (sql/do-nothing))
          (sql/returning :*))
       (assoc model :datasources)))

(s/fdef insert-datasources!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

(defn update-model!
  "Update the data sources and the latest reference time of `model`."
  [db model]
  (->> (assoc model :datasources (dods/datasources model))
       (update-latest-reference-time! db)
       (insert-datasources! db)))

(s/fdef update-model!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

(defmulti process
  (fn [message] (-> message :routing-key keyword)))

(defmethod process :weather.model.update
  [{:keys [body channel env]}]
  (let [model (update-model! (:db env) (:model body))]
    (->> {:body {:model model}
          :exchange "worker"
          :routing-key "weather.model.updated"}
         (rabbitmq/publish-edn channel))
    (log/info {:msg "Updated weather model inventory."
               :model (update model :datasources count)})
    {:status :ack}))

(defn worker
  "Return a worker that updates the weather inventory."
  [config]
  (-> process
      (k/consumer
       {:as rabbitmq/read-edn
        :consumer-name "Weather Models"})
      (dead-letter/with-durable-dead-letter-backoff)
      (k/with-durable-queue "worker.weather.models"
        {:exchange "worker"
         :routing-keys ["weather.model.update"]})
      (k/with-channel {:prefetch-count 1})
      (k/with-connection (rabbitmq/config (:broker config)))
      (k/with-env)
      (component/using [:db :broker :topology])))

;; Client

(defrecord Client [broker db]
  component/Lifecycle
  (start [client]
    (rabbitmq/with-channel [channel broker]
      (let [models (models/all db)]
        (doseq [model models]
          (->> {:body {:model model}
                :exchange "worker"
                :routing-key "weather.model.update"}
               (rabbitmq/publish-edn channel))
          (log/info {:msg "Queued weather model to update inventory."
                     :model model})))
      client))
  (stop [client]
    client))

(defn publish-update-model!
  "Publish commands to update the data sources of the weather models."
  [{:keys [db broker] :as config}]
  (-> (component/system-map
       :broker (rabbitmq/client broker)
       :client (map->Client {})
       :db (db/new-db db)
       :topology (topology/topology broker))
      (component/system-using
       {:client [:broker :db :topology]
        :topology [:broker]})
      (component/start)
      (component/stop)))

(defn -main
  "Publish commands to update the data sources of the weather models."
  [& args]
  (publish-update-model! (config env)))

(comment
  (-main))
