(ns burningswell.worker.storage
  (:require [clojure.tools.logging :as log]
            [clojure.walk :as walk]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]]
            [clojure.spec.alpha :as s])
  (:import [com.google.cloud.storage Acl Acl$Role Acl$User BlobInfo
            Storage$BlobTargetOption StorageOptions]))

(s/def ::acl (s/nilable vector?))
(s/def ::cache-control (s/nilable string?))
(s/def ::content-encoding (s/nilable keyword?))
(s/def ::key string?)
(s/def ::md5 (s/nilable string?))
(s/def ::meta-data (s/nilable map?))

(s/def ::blob
  (s/keys :req-un [::key]
          :opt-un [::acl
                   ::cache-control
                   ::content-encoding
                   ::md5
                   ::meta-data]))

(def acl-public
  [(Acl/of (Acl$User/ofAllUsers) Acl$Role/READER)])

(defn acl
  "Returns the access control list for `blob`."
  [blob]
  (or (:acl blob) acl-public))

(defn cache-control
  "Returns the cache control for `blob`."
  [blob]
  (or (:cache-control blob) "public, max-age 31536000"))

(defn content-encoding
  "Returns the content encoding for `blob`."
  [blob]
  (name (or (:content-encoding blob) :identity)))

(defn md5
  "Return the MD5 sum for `blob`."
  [blob]
  (some-> blob :bytes digest/md5-bytes base64-encode))

(defn meta-data
  "Return the MD5 sum for `blob`."
  [blob]
  (walk/stringify-keys (:meta-data blob)))

(defprotocol Storage
  (-save! [storage blob]))

(s/def ::storage #(satisfies? Storage %))

(defn- blob-info
  "Returns the `blob` info as a map."
  [blob]
  {:acl (acl blob)
   :cache-control (cache-control blob)
   :content-disposition (:content-disposition blob)
   :content-encoding (content-encoding blob)
   :content-type (:content-type blob)
   :key (:key blob)
   :md5 (md5 blob)
   :meta-data (meta-data blob)})

(defn- make-blob-info
  "Returns the `blob` info for `storage`."
  [{:keys [bucket] :as storage} blob]
  (let [info (blob-info blob)]
    (-> (BlobInfo/newBuilder bucket (:key info))
        (.setAcl (:acl info))
        (.setCacheControl (:cache-control info))
        (.setContentDisposition (:content-disposition info))
        (.setContentEncoding (:content-encoding info))
        (.setContentType (:content-type info))
        (.setMd5 (:md5 info))
        (.setMetadata (:meta-data info))
        (.build))))

(defn- google-storage-url
  "Return the `storage` url for `blob`."
  [storage blob]
  (format "https://storage.googleapis.com/%s/%s" (:bucket storage) (:key blob)))

(defrecord GoogleStorage [bucket]
  Storage
  (-save! [storage blob]
    (.create (.. StorageOptions getDefaultInstance getService)
             (make-blob-info storage blob)
             (:bytes blob)
             (into-array Storage$BlobTargetOption []))
    (assoc blob :url (google-storage-url storage blob))))

(defn save! [storage blob]
  (let [result (-save! storage blob)]
    (log/infof "Saved \"%s\" to Google Cloud Storage." (:url result))
    result))

(s/fdef save!
  :args (s/cat :storage ::storage :blob ::blob)
  :ret ::blob)

(defn google
  "Returns a new Google Cloud storage."
  [& [opts]]
  (map->GoogleStorage opts))
