(ns burningswell.worker.weather.datasources
  (:require [burningswell.db.weather.datasources :as datasources]
            [burningswell.db.weather.variables :as variables]
            [burningswell.rabbitmq.client :as rabbitmq]
            [burningswell.worker.config :refer [config]]
            [burningswell.worker.util :as util]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [kithara.core :as k]
            [kithara.patterns.dead-letter-backoff :as dead-letter]
            [kithara.patterns.threads :refer [with-threads]]
            [netcdf.dataset :as dataset]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(defn- datasets
  "Fetch the datasets of the `datasource` and `variables`."
  [datasource variables]
  (try (dataset/with-grid-dataset [grid (:dods datasource)]
         (->> (for [valid-time (dataset/valid-times grid)
                    variable variables]
                {:datasource-id (:id datasource)
                 :valid-time valid-time
                 :variable-id (:id variable)})
              doall not-empty))
       (catch Throwable e
         (throw (ex-info (str "Can't read datasource: " (.getMessage e))
                         {:type ::read-datasource-error
                          :datasource (select-keys datasource [:id :dods])}
                         e)))))

(defn update-datasource!
  "Update the datasets of the `datasource`."
  ([db datasource]
   (->> (variables/by-datasource db datasource)
        (update-datasource! db datasource)))
  ([db datasource variables]
   (if-let [datasets (datasets datasource variables)]
     (->> @(sql/insert db :weather.datasets []
             (sql/values datasets)
             (sql/on-conflict [:datasource-id :variable-id :valid-time]
               (sql/do-nothing))
             (sql/returning :*))
          (assoc datasource :datasets))
     (assoc datasource :datasets []))))

(s/def ::update-datasource-model
  (s/cat :db sql/db?
         :datasource ::datasources/datasource))

(s/def ::update-datasource-model-variables
  (s/cat :db sql/db?
         :datasource ::datasources/datasource
         :variables (s/coll-of ::variables/variable)))

(s/fdef update-datasource!
  :args (s/or :model ::update-datasource-model
              :model+variables ::update-datasource-model-variables))

(defmulti process
  (fn [message] (-> message :routing-key keyword)))

(defmethod process :weather.datasource.update
  [{:keys [body channel env]}]
  (try+ (let [datasource (update-datasource! (:db env) (:datasource body))]
          (->> {:body {:datasource datasource}
                :exchange "worker"
                :routing-key "weather.datasource.updated"}
               (rabbitmq/publish-edn channel))
          (log/info {:msg "Updated the weather datasource."
                     :datasource (update datasource :datasets count)}))
        (catch [:type ::read-datasource-error] {:keys [datasource]}
          (log/warn {:msg (:message &throw-context)
                     :datasource datasource}))))

(defmethod process :weather.model.updated
  [{:keys [body channel env]}]
  (doseq [datasource (-> body :model :datasources)]
    (->> {:body {:datasource datasource}
          :exchange "worker"
          :routing-key "weather.datasource.update"}
         (rabbitmq/publish-edn channel)))
  {:status :ack})

(defn worker
  "Return a worker that updates the weather inventory."
  [{:keys [threads] :as config}]
  (let [threads (or threads (util/available-processors))]
    (-> (k/consumer process {:as rabbitmq/read-edn
                             :consumer-name "Weather Data Sources"})
        (dead-letter/with-durable-dead-letter-backoff)
        (k/with-durable-queue "worker.weather.datasources"
          {:exchange "worker"
           :routing-keys ["weather.datasource.update"
                          "weather.model.updated"]})
        (with-threads threads)
        (k/with-channel {:prefetch-count threads})
        (k/with-connection (rabbitmq/config (:broker config)))
        (k/with-env)
        (component/using [:db :broker :topology]))))
