(ns burningswell.worker.topics
  (:gen-class)
  (:require [burningswell.edn :as edn]
            [burningswell.worker.config :as config]
            [environ.core :refer [env]]
            [jackdaw.admin :as admin]
            [jackdaw.serdes.edn :as jse]
            [jackdaw.serdes.json :as json]
            [taoensso.timbre :as log]))

(defonce registry {})

(defn config
  "Make a topic config."
  ([topic-name]
   (config topic-name {}))
  ([topic-name & [{:keys [key-serde value-serde]}]]
   {:topic-name topic-name
    :partition-count 1
    :replication-factor 1
    :topic-config {}
    :key-serde (or key-serde (jse/serde edn/read-opts))
    :value-serde (or value-serde (jse/serde edn/read-opts))}))

(defmacro deftopic [sym name & opts]
  `(do (def ~sym (config ~name ~(apply hash-map (seq opts))))
       (alter-var-root #'registry assoc ~(keyword sym) ~sym)))

(deftopic api-commands-edn
  "burningswell.api.commands")

(deftopic api-commands-json
  "burningswell.api.commands.json"
  :key-serde (json/serde)
  :value-serde (json/serde))

(deftopic api-events-edn
  "burningswell.api.events")

(deftopic api-events-json
  "burningswell.api.events.json"
  :key-serde (json/serde)
  :value-serde (json/serde))

(deftopic page-views-total
  "burningswell.page-views.urls.total")

(deftopic page-views-total-per-user
  "burningswell.page-views.urls.total-per-user")

(deftopic spot-photo-created
  "burningswell.spots.photos.created")

(deftopic user-profile-photo-created
  "burningswell.users.profile-photos.created")

(deftopic weather-dataset-update
  "burningswell.weather.dataset.update")

(deftopic weather-dataset-update-succeeded
  "burningswell.weather.dataset.update-succeeded")

(deftopic weather-datasource-update-succeeded
  "burningswell.weather.datasource.update-succeeded")

(deftopic weather-model-update
  "burningswell.weather.model.update")

(deftopic weather-model-update-succeeded
  "burningswell.weather.model.update-succeeded")

(deftopic weather-spots-update-succeeded
  "burningswell.weather.spots.update-succeeded")

(defn- admin-config [env]
  (let [{:keys [kafka]} (config/config env)]
    {"bootstrap.servers" (:bootstrap.servers kafka)}))

(defn create-or-alter-topics!
  ([config]
   (create-or-alter-topics! config (vals registry)))
  ([config topics]
   (with-open [admin (admin/->AdminClient
                      {"bootstrap.servers"
                       (:bootstrap.servers config)})]
     (doseq [topic topics]
       ;; TODO: Try topic-exists?
       (try (admin/alter-topic-config! admin [topic])
            (log/info {:msg "Updated topic." :topic topic} )
            (catch Exception e
              (admin/create-topics! admin [topic])
              (admin/retry-exists? admin topic 10 100)
              (log/info {:msg "Created topic." :topic topic} )))))))

(defn -main [& args]
  (create-or-alter-topics! (:kafka (config/config env))))

(comment (-main))
