(ns burningswell.services.flickr
  (:require [burningswell.db.flickr.photo-sizes :as photo-sizes]
            [burningswell.db.flickr.photos :as photos]
            [burningswell.db.flickr.users :as users]
            [burningswell.specs.core :as core]
            [clj-http.client :as http]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [inflections.core :as infl]
            [no.en.core :refer [parse-integer]]
            [sqlingvo.core :as sql]
            [com.stuartsierra.component :as component]))

;; User

(s/def :flickr.user/nsid string?)
(s/def :flickr.user/id string?)
(s/def :flickr.user/location string?)
(s/def :flickr.user/realname string?)
(s/def :flickr.user/username string?)

(s/def :flickr/user
  (s/keys :req-un [:flickr.user/id
                   :flickr.user/location
                   :flickr.user/nsid
                   :flickr.user/realname
                   :flickr.user/username]))

;; Owner

(s/def :flickr/owner
  (s/keys :req-un [:flickr.user/location
                   :flickr.user/nsid
                   :flickr.user/realname
                   :flickr.user/username]))

;; Photo size

(s/def :flickr.photo.size/height nat-int?)
(s/def :flickr.photo.size/label string?)
(s/def :flickr.photo.size/media string?)
(s/def :flickr.photo.size/source ::core/url-str)
(s/def :flickr.photo.size/url string?)
(s/def :flickr.photo.size/width nat-int?)

(s/def :flickr.photo/size
  (s/keys :req-un [:flickr.photo.size/height
                   :flickr.photo.size/label
                   :flickr.photo.size/media
                   :flickr.photo.size/source
                   :flickr.photo.size/url
                   :flickr.photo.size/width]))

;; Photo

(s/def :flickr.photo/farm pos-int?)
(s/def :flickr.photo/id string?)
(s/def :flickr.photo/isfamily #{0 1})
(s/def :flickr.photo/isfriend #{0 1})
(s/def :flickr.photo/ispublic #{0 1})
(s/def :flickr.photo/owner string?)
(s/def :flickr.photo/secret string?)
(s/def :flickr.photo/server string?)
(s/def :flickr.photo/title string?)
(s/def :flickr.photo/url ::core/url-str)

(s/def :flickr/photo-without-url
  (s/keys :req-un [:flickr.photo/farm
                   :flickr.photo/id
                   :flickr.photo/isfamily
                   :flickr.photo/isfriend
                   :flickr.photo/ispublic
                   :flickr.photo/owner
                   :flickr.photo/secret
                   :flickr.photo/server
                   :flickr.photo/title]))

(s/def :flickr/photo
  (s/merge :flickr/photo-without-url
           (s/keys :req-un [:flickr.photo/url])))

;; Photo Info

(s/def :flickr.photo.info/id :flickr.photo/id)
(s/def :flickr.photo.info/owner :flickr/owner)

(s/def :flickr.photo/info
  (s/keys :req-un [:flickr.photo/id
                   :flickr.photo.info/owner]))

;; User Info

(s/def :flickr.user/info
  (s/keys :req-un [:flickr.user/id
                   :flickr.user/nsid]))

(s/def :flickr.user/info
  (s/keys :req-un [:flickr.user/id
                   :flickr.user/nsid]))

(s/def ::service any?)

(s/def ::width pos-int?)
(s/def ::height pos-int?)

(s/def ::label
  #{"Large" "Medium" "Original" "Small" "Square" "Thumbnail"})

(s/def ::size (s/keys :req-un [::width ::height ::label]))

(def licenses
  "The Flickr license types."
  [{:id 0
    :name "All Rights Reserved"
    :usable? false}
   {:id 1
    :name "Attribution-NonCommercial-ShareAlike License"
    :url "http://creativecommons.org/licenses/by-nc-sa/2.0/"
    :usable? false}
   {:id 2
    :name "Attribution-NonCommercial License"
    :url "http://creativecommons.org/licenses/by-nc/2.0/"
    :usable? false}
   {:id 3
    :name "Attribution-NonCommercial-NoDerivs License"
    :url "http://creativecommons.org/licenses/by-nc-nd/2.0/"
    :usable? false}
   {:id 4
    :name "Attribution License"
    :url "http://creativecommons.org/licenses/by/2.0/"
    :usable? true}
   {:id 5
    :name "Attribution-ShareAlike License"
    :url "http://creativecommons.org/licenses/by-sa/2.0/"
    :usable? true}
   {:id 6
    :name "Attribution-NoDerivs License"
    :url "http://creativecommons.org/licenses/by-nd/2.0/"
    :usable? true}
   {:id 7
    :name "No known copyright restrictions"
    :url "http://flickr.com/commons/usage/"
    :usable? true}
   {:id 8
    :name "United States Government Work"
    :url "http://www.usa.gov/copyright.shtml"
    :usable? true}])

(defrecord Flickr [api-key client-id client-secret])

(defn client
  "Returns a Flickr client using `api-key`."
  [{:keys [api-key client-id client-secret] :as opts}]
  (-> (map->Flickr opts)
      (component/using [:db])))

(defn photo-url
  "Returns the Flickr URL of `photo`."
  [photo & [size]]
  (str "http://farm" (:farm photo) ".staticflickr.com/"
       (:server photo) "/" (:id photo) "_" (:secret photo)
       (if size (str "_" (name size)))
       ".jpg"))

(s/fdef photo-url
  :args (s/cat :photo :flickr/photo-without-url)
  :ret ::core/url-str)

(defn- strip-content [x]
  (let [content (:_content x)]
    (when-not (str/blank? content)
      content)))

(defn request
  "Build a Flickr request."
  [service method params]
  (http/request
   {:url "https://api.flickr.com/services/rest"
    :as :auto
    :method :get
    :accept "application/json"
    :query-params
    (assoc (infl/underscore-keys params)
           :api_key (:api-key service)
           :method (name method)
           :format "json"
           :nojsoncallback "1")}))

(defn photo-info
  "Return information about a photo."
  [service photo-id & [opts]]
  (-> (request service :flickr.photos.getInfo (assoc opts :photo_id photo-id))
      :body :photo infl/hyphenate-keys))

(s/fdef photo-info
  :args (s/cat :service ::service :photo-id :flickr.photo/id)
  :ret (s/nilable :flickr.photo/info))

(defn photo-sizes
  "Request a image sizes of `photo-id`."
  [service photo-id & [opts]]
  (->> (request service :flickr.photos.getSizes (assoc opts :photo_id photo-id))
       :body :sizes :size
       (map (fn [size]
              (-> (infl/hyphenate-keys size)
                  (update :width parse-integer)
                  (update :height parse-integer))))))

(s/fdef photos-sizes
  :args (s/cat :service ::service :photo-id :flickr.photo/id)
  :ret (s/nilable (s/coll-of :flickr.photo/size)))

(s/def :flickr.search-photos.opts/text string?)

(s/def :flickr.search-photos/opts
  (s/keys :opt-un [:flickr.search-photos.opts/text]))

(defn search-photos
  "Request a list of photos matching some criteria."
  [service opts]
  (->> (request service :flickr.photos.search opts)
       :body :photos :photo
       (map (fn [photo]
              (-> (infl/hyphenate-keys photo)
                  (assoc :url (photo-url photo)))))))

(s/fdef search-photos
  :args (s/cat :service ::service :opts :flickr.search-photos/opts)
  :ret (s/nilable (s/coll-of :flickr/photo)))

(defn user-info
  "Return information about a user."
  [service user-id & [opts]]
  (-> (request service :flickr.people.getInfo (assoc opts :user_id user-id))
      :body :person infl/hyphenate-keys))

(s/fdef user-info
  :args (s/cat :service ::service :user-id :flickr.user/id)
  :ret (s/nilable :flickr.user/info))

(defn save-photo-sizes!
  "Save the `sizes` of `photo` to `db`."
  [db photo sizes]
  (->> (map #(assoc % :photo-id (:id photo)) sizes)
       (map #(photo-sizes/save! db %))
       (doall)))

(s/fdef save-photo-sizes!
  :args (s/cat :db sql/db?
               :photo :flickr/photo
               :sizes (s/coll-of :flickr.photo/size))
  :ret (s/nilable :flickr.user/info))

(defn save-photo!
  "Save the `photo` to `db`."
  [db photo]
  (users/save! db {:id (:owner photo)})
  (photos/save! db photo))

(s/fdef save-photo!
  :args (s/cat :db sql/db? :photo :flickr/photo)
  :ret ::photos/photo)

(defn save-user-info!
  "Save the `user-info` to `db`."
  [db user-info]
  (->> {:id (:id user-info)
        :nsid (:nsid user-info)
        :username (-> user-info :username :-content )
        :realname (-> user-info :realname :-content )
        :location (-> user-info :location :-content )
        :profile-url (-> user-info :profileurl :-content)
        :photos-url (-> user-info :photosurl :-content)}
       (users/save! db)))

(s/fdef save-user-info!
  :args (s/cat :db sql/db? :user-info :flickr.user/info)
  :ret ::users/user)
