(ns burningswell.worker.driver
  (:require [jackdaw.streams :as j]
            [taoensso.timbre :as log]))

(defprotocol Application
  (config [app]
    "Returns the Kafka Streams config for `app`.")
  (topology [app builder]
    "Returns the Kafka Streams topology for `app`."))

(defprotocol IDriver
  (start [driver app]
    "Start the Kafka Streams `app`.")
  (stop [driver app]
    "Stop the Kafka Streams `app`."))

(defn- app-id
  "Returns the `app` id."
  [app]
  (get (config app) "application.id"))

(defn- start-app
  "Start the `app`."
  [driver app]
  (let [builder (j/streams-builder)
        topology (topology app builder)
        stream (j/kafka-streams builder (config app))]
    (j/start stream)
    (log/info {:msg (format "Application %s successfully started." (app-id app))
               :config (config app)})
    (assoc app :stream stream)))

(defn- stop-app
  "Stop the `app`."
  [driver app]
  (when-let [stream (:stream app)]
    (j/close stream)
    (log/info {:msg (format "Application %s successfully stopped." (app-id app))
               :config (config app)}))
  (assoc app :stream nil))

(defrecord Driver []
  IDriver
  (start [driver app]
    (start-app driver app))
  (stop [driver app]
    (stop-app driver app)))

(defn driver
  "Returns a new test driver."
  []
  (map->Driver {}))
