(ns burningswell.services.gravatar
  (:refer-clojure :exclude [test])
  (:require [burningswell.db.emails :as emails]
            [burningswell.db.gravatar.images :as images]
            [burningswell.digest :refer [md5-sum]]
            [clj-http.client :as http]
            [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.string :as str]
            [datumbazo.core :as sql]
            [inflections.core :as inf]
            [burningswell.db.gravatar.users :as users]
            [com.stuartsierra.component :as component]))

(defprotocol Service
  (-profile [service user opts]))

(s/def ::address string?)
(s/def ::email (s/keys :req-un [::address]))

(s/def ::user-agent string?)
(s/def ::url string?)

(s/def ::config
  (s/keys :req-un [::url ::user-agent]))

(s/def ::service #(satisfies? Service %))

(def default-config
  "The Gravatar default config."
  {:url "https://www.gravatar.com"
   :user-agent "Burning Swell"})

(defn- headers
  "Returns the HTTP request headers for `service`."
  [service]
  {"User-Agent" (-> service :config :user-agent)})

(defn email-hash
  "Returns the Gravatar hash code for `email`."
  [{:keys [address] :as email}]
  (some-> address str/trim str/lower-case md5-sum))

(s/fdef email-hash
  :args (s/cat :email ::email)
  :ret (s/nilable string?))

(defn avatar-url
  "Returns the Gravatar avatar url for `email`."
  [service email]
  (some->> email email-hash (str (-> service :config :url) "/avatar/")))

(s/fdef avatar-url
  :args (s/cat :service ::service :email ::email)
  :ret (s/nilable string?))

(defn avatar
  "Returns the Gravatar avatar as a byte array."
  [service email]
  (let [{:keys [status body]}
        (http/get (avatar-url service email)
                  {:as :byte-array
                   :headers (headers service)
                   :throw-exceptions? false})]
    (when (= status 200) body)))

(s/fdef avatar-url
  :args (s/cat :service ::service :email ::email)
  :ret (s/nilable bytes?))

(defn profile-url
  "Returns the Gravatar profile url for `email`."
  [service email]
  (some->> email email-hash (str (-> service :config :url) "/")))

(s/fdef profile-url
  :args (s/cat :service ::service :email ::email)
  :ret (s/nilable string?))

(defn profile
  "Returns the Gravatar profile for `email`."
  [service email & [opts]]
  (-profile service email opts))

(s/fdef profile
  :args (s/cat :service ::service
               :email ::email
               :opts (s/? (s/nilable map?)))
  :ret (s/nilable :gravatar/profile))

(defn- fetch-profile
  "Fetch the Gravatar profile."
  [service email & [opts]]
  (let [url (str (profile-url service email) ".json")
        {:keys [status body]}
        (http/get url (merge
                       {:as :json
                        :headers (headers service)
                        :throw-exceptions? false}
                       opts))]
    (cond
      (= status 200) (-> (inf/hyphenate-keys body) :entry first)
      (= status 404) nil
      :else (throw (ex-info (str "Can't fetch Gravatar profile: " status)
                            {:email email :url url :opts opts})))))

(defn exists?
  "Returns true if the Gravatar profile for `email` exists, otherwise false."
  [service email]
  (some? (profile service email)))

(defrecord Gravatar [config]
  Service
  (-profile [service email opts]
    (fetch-profile service email opts)))

(defrecord Test [config]
  Service
  (-profile [service email opts]
    (when (re-matches #".*burningswell.*" (str (:address email)))
      (gen/generate (s/gen :gravatar/profile)))))

(defn service
  [& [opts]]
  (-> (map->Gravatar {:config (merge default-config opts)})
      (component/using [:db])))

(defn test
  [& [opts]]
  (map->Test {}))

;; Synchronize user

(defn- save-images!
  "Save the images of the Gravatar `profile`."
  [db profile]
  (when-let [images (seq (:photos profile))]
    (some->> (seq (:photos profile))
             (map #(assoc % :user-id (:id profile)) )
             (images/save-all! db))))

(defn synchronize-user!
  "Synchronize the Gravatar information for `user`."
  [{:keys [db] :as service} user & [opts]]
  (doall (for [email (emails/by-user db user)
               :let [profile (profile service email)]
               :when profile]
           (do (users/save! db profile)
               (save-images! db profile)
               (emails/update-gravatar! db email (:id profile))
               profile))))
