(ns burningswell.worker.page-views
  (:require [burningswell.api.specs.events]
            [burningswell.routes :as routes]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.topics :as topics]
            [com.stuartsierra.component :as component]
            [datumbazo.io :refer [jsonb]]
            [jackdaw.streams :as j]
            [no.en.core :refer [format-url parse-integer parse-url]]
            [sqlingvo.core :as sql]
            [inflections.core :as infl]))

(defn config
  "Returns the config for the page views app."
  [& [opts]]
  (->> {:application
        {"application.id" "page-views"
         ;; "auto.offset.reset" "earliest"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"}
        :routes #{:continent :country :photo :region :spot :user}
        :input {:events topics/api-events-edn}
        :stores {:route-totals (topics/config "page-views.route-totals")
                 :route-viewer-totals (topics/config "page-views.route-viewer-totals")
                 :url-totals (topics/config "page-views.url-totals")
                 :url-viewer-totals (topics/config "page-views.url-viewer-totals")}}
       (merge opts)))

(defn strip-url
  "Strip the query params from `url`."
  [url]
  (some-> (parse-url url)
          (dissoc :query-params)
          (format-url)
          (java.net.URL.)))

(defn match-route
  "Returns the matching route for `url`, or nil. "
  [url]
  (when-let [{:keys [handler route-params]} (routes/match-url url)]
    {:name handler :params route-params}))

(defmulti normalize-route
  "Normalize the given `route`."
  (fn [route] (:name route)))

(defmethod normalize-route :default
  [{:keys [name params] :as route}]
  (cond-> route
    (seq params)
    (update :params select-keys [:id])
    (:id params)
    (update :params #(update % :id parse-integer))))

(defn- row-exists?
  "Returns true if the row with `id` exists in `table`."
  [db table id]
  (not (empty? @(sql/select db [1]
                  (sql/from table)
                  (sql/where `(= :id ~id))))))

(defn- viewer-exists? [db viewer-id]
  (row-exists? db :users viewer-id))

(defn- route-exists? [db route]
  (let [id (-> route :params :id)
        table (-> route :name infl/plural)]
    (or (nil? id) (row-exists? db table id))))

;; Page view events

(defn- page-view-event?
  "Returns true if the `event` is a page view event, otherwise false."
  [[key event]]
  (= (:name event) :burningswell.api.events/page-view-created))

(defn- page-view-events
  "Build a stream of page view events."
  [app builder]
  (-> (j/kstream builder (-> app :config :input :events))
      (j/filter page-view-event?)))

;; URL totals

(defn url-total-key
  "Returns the key for the total page counts."
  [[key v]]
  (dissoc key :user-id))

(defn- url-totals
  "Build a table of total page views."
  [app url-viewer-totals]
  (-> url-viewer-totals
      (j/to-kstream)
      (j/group-by url-total-key (topics/config "total"))
      (j/count (-> app :config :stores :url-totals))))

(defn- url-row [key total]
  (let [{:keys [scheme server-name server-port uri]} (parse-url (:url key))
        route (:route key)]
    {:route-params (some-> route :params jsonb)
     :path (or uri "/")
     :route (some-> route :name name)
     :scheme (name scheme)
     :server-name server-name
     :server-port server-port
     :total total
     :url (-> key :url str)}))

(defn save-url-total!
  "Save the page view to the database."
  [{:keys [db] :as app} url total]
  @(sql/insert db :page-views.urls []
     (sql/values [(url-row url total)])
     (sql/on-conflict [:url]
       (sql/do-update {:total :EXCLUDED.total}))))

(defn save-url-totals!
  "Save the page view to the database."
  [app url-totals]
  (-> (j/to-kstream url-totals)
      (j/peek (fn [[url total]] (save-url-total! app url total)))))

;; URL viewer totals

(defn url-viewer-total-key
  "Returns the key for the total page counts per user."
  [[key {:keys [url user-id]}]]
  {:route (match-route url)
   :url (strip-url url)
   :user-id user-id})

(defn- url-viewer-totals
  "Build a table of total page views per user."
  [app events]
  (-> (j/group-by events url-viewer-total-key (topics/config "total-per-viewer"))
      (j/count (-> app :config :stores :url-viewer-totals))))

(defn save-url-viewer-total!
  "Save the page view by user to the database."
  [{:keys [db] :as app} url total]
  (when (viewer-exists? db (:user-id url))
    @(sql/insert db :page-views.urls-by-viewer []
       (sql/values [(assoc (url-row url total) :viewer-id (:user-id url))])
       (sql/on-conflict [:url :viewer-id]
         (sql/do-update {:total :EXCLUDED.total})))))

(defn save-url-viewer-totals!
  "Save the page view to the database."
  [app url-viewer-totals]
  (-> (j/to-kstream url-viewer-totals)
      (j/peek (fn [[url total]] (save-url-viewer-total! app url total)))))

;; Route totals

(defn- route-totals-key
  [[k v]]
  {:route (:route k)})

(defn- route-totals
  "Calculate the total route views per route."
  [app route-viewer-totals]
  (-> (j/to-kstream route-viewer-totals)
      (j/group-by route-totals-key (topics/config "total"))
      (j/count (-> app :config :stores :route-totals))))

(defn- route-row [route total]
  {:name (-> route :name name)
   :params (-> route :params jsonb)
   :total total})

(defn- route-table
  [route]
  (keyword (str "page-views." (infl/plural (-> route :name name)))))

(defn- route-table-viewer
  [route]
  (keyword (str (name (route-table route)) "-by-viewer")))

(defn- route-foreign-key
  [route]
  (keyword (str (-> route :name name) "-id")))

(defn- save-table?
  [app route]
  (contains? (-> app :config :routes) (:name route)))

(defn save-route-total-table!
  "Save the page view to the database."
  [{:keys [db] :as app} {:keys [route] :as key} total]
  (when (and (save-table? app route)
             (route-exists? db route))
    @(sql/insert db (route-table route) []
       (sql/values [{(route-foreign-key route) (-> route :params :id)
                     :total total}])
       (sql/on-conflict [(route-foreign-key route)]
         (sql/do-update {:total :EXCLUDED.total})))))

(defn save-route-total!
  "Save the page view to the database."
  [{:keys [db] :as app} {:keys [route] :as key} total]
  @(sql/insert db :page-views.routes []
     (sql/values [(route-row (:route key) total)])
     (sql/on-conflict [:name :params]
       (sql/do-update {:total :EXCLUDED.total}))))

(defn save-route-totals!
  "Save the page view to the database."
  [app route-totals]
  (-> (j/to-kstream route-totals)
      (j/peek (fn [[key total]]
                (save-route-total! app key total)
                (save-route-total-table! app key total)))))

;; Route totals by viewer

(defn- route-viewer-totals-key
  [[k v]]
  {:route (normalize-route (:route k))
   :user-id (:user-id k)})

(defn- route-viewer-totals
  "Calculate the total route views per route by viewer."
  [app url-viewer-totals]
  (-> (j/to-kstream url-viewer-totals)
      (j/filter (fn [[k v]] (:route k)))
      (j/group-by route-viewer-totals-key (topics/config "route-totals-by-viewer"))
      (j/count (-> app :config :stores :route-viewer-totals))))

(defn save-route-viewer-total-table!
  "Save the page view to the database."
  [{:keys [db] :as app} {:keys [route] :as key} total]
  (when (and (save-table? app route)
             (route-exists? db route)
             (viewer-exists? db (:user-id key)))
    @(sql/insert db (route-table-viewer route) []
       (sql/values [{(route-foreign-key route) (-> route :params :id)
                     :total total
                     :viewer-id (:user-id key)}])
       (sql/on-conflict [(route-foreign-key route) :viewer-id]
         (sql/do-update {:total :EXCLUDED.total})))))

(defn save-route-viewer-total!
  "Save the page view to the database."
  [{:keys [db] :as app} key total]
  (when (viewer-exists? db (:user-id key))
    @(sql/insert db :page-views.routes-by-viewer []
       (sql/values [(assoc (route-row (:route key) total) :viewer-id (:user-id key))])
       (sql/on-conflict [:name :params :viewer-id]
         (sql/do-update {:total :EXCLUDED.total})))))

(defn save-route-viewer-totals!
  "Save the page view to the database."
  [app route-viewer-totals]
  (-> (j/to-kstream route-viewer-totals)
      (j/peek (fn [[route total]]
                (save-route-viewer-total! app route total)
                (save-route-viewer-total-table! app route total)))))

(defn- build-topology
  "Build the page view topology."
  [app builder]
  (let [page-views (page-view-events app builder)
        url-viewer-totals (url-viewer-totals app page-views)
        url-totals (url-totals app url-viewer-totals)
        route-viewer-totals (route-viewer-totals app url-viewer-totals)
        route-totals (route-totals app route-viewer-totals)]
    (save-url-viewer-totals! app url-viewer-totals)
    (save-url-totals! app url-totals)
    (save-route-totals! app route-totals)
    (save-route-viewer-totals! app route-viewer-totals)))

(defrecord PageViews [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn page-views [& [opts]]
  (-> (map->PageViews {:config (config opts)})
      (component/using [:db :driver])))
