(ns burningswell.worker.weather.datasources
  (:require [burningswell.db.weather.datasources :as datasources]
            [burningswell.db.weather.models :as models]
            [burningswell.db.weather.variables :as variables]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.topics :as topics]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [datumbazo.core :as sql]
            [jackdaw.streams :as j]
            [netcdf.dataset :as dataset]
            [taoensso.timbre :as log]))

(defn config
  "Returns the configuration for the app."
  [& [opts]]
  (->> {:application
        {"application.id" "update-weather-datasources"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"}
        :input {:commands topics/weather-model-update-succeeded}
        :output {:succeeded topics/weather-datasource-update-succeeded}}
       (merge opts)))

(defn- datasets
  "Fetch the datasets of the `datasource` and `variables`."
  [datasource variables]
  (try (dataset/with-grid-dataset [grid (:dods datasource)]
         (->> (for [valid-time (dataset/valid-times grid)
                    variable variables]
                {:datasource-id (:id datasource)
                 :valid-time valid-time
                 :variable-id (:id variable)})
              doall not-empty))
       (catch Throwable e
         (throw (ex-info (str "Can't read datasource: " (.getMessage e))
                         {:type ::read-datasource-error
                          :datasource (select-keys datasource [:id :dods])}
                         e)))))

(defn- insert-datasets! [db datasource datasets]
  (->> @(sql/insert db :weather.datasets []
          (sql/values datasets)
          (sql/on-conflict [:datasource-id :variable-id :valid-time]
            (sql/do-nothing))
          (sql/returning :*))
       (assoc datasource :datasets)))

(defn update-datasource!
  "Update the datasets of the `datasource`."
  ([db datasource]
   (->> (variables/by-datasource db datasource)
        (update-datasource! db datasource)))
  ([db datasource variables]
   (log/info {:msg "Updating weather datasource."
              :datasource datasource})
   (let [datasource (merge datasource (datasources/by-id db (:id datasource)))] ;; TODO: Remove after https switch
     (if-let [datasets (datasets datasource variables)]
       (let [{:keys [datasets model-id] :as datasource}
             (insert-datasets! db datasource datasets)
             valid-times (sort (map :valid-time datasets))
             model (models/by-id db model-id)]
         (log/info {:msg "Updated weather datasource."
                    :model (select-keys model [:id :name])
                    :reference-time (:reference-time datasource)
                    :variables (mapv :name variables)
                    :start (first valid-times)
                    :end (last valid-times)})
         datasource)
       (assoc datasource :datasets [])))))

(s/def ::update-datasource-model
  (s/cat :db sql/db?
         :datasource ::datasources/datasource))

(s/def ::update-datasource-model-variables
  (s/cat :db sql/db?
         :datasource ::datasources/datasource
         :variables (s/coll-of ::variables/variable)))

(s/fdef update-datasource!
  :args (s/or :model ::update-datasource-model
              :model+variables ::update-datasource-model-variables))

;; Kafka Streams

(defn build-topology
  "Build the Kafka Streams topology."
  [{:keys [db config] :as app} builder]
  (-> (j/kstream builder (-> config :input :commands))
      (j/flat-map-values #(:datasources %))
      (j/map-values #(update-datasource! db %1))
      (j/to (-> config :output :succeeded))))

(defrecord Worker [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn worker
  "Returns a new weather datasource worker."
  [& [opts]]
  (-> (map->Worker {:config (config opts)})
      (component/using [:db :driver])))
