(ns burningswell.services.photos
  (:require [again.core :as again]
            [burningswell.db.photos :as photos]
            [burningswell.db.users :as users]
            [burningswell.services.storage :as storage]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [datumbazo.core :as sql]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]])
  (:import java.awt.image.BufferedImage
           [java.io ByteArrayInputStream ByteArrayOutputStream]
           javax.imageio.ImageIO
           org.apache.commons.io.IOUtils))

(def user-agent
  "The user agent string."
  (str "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 "
       "(KHTML, like Gecko) Chrome/66.0.3359.117 Safari/537.36"))

(defprotocol Service
  (-import! [service photo opts]))

(defn import!
  "Create the `photo`."
  [service photo & [opts]]
  (merge photo (-import! service photo opts)))

(defn- buffered-image->bytes
  "Convert `buffered-image` into a byte in `format`."
  [^BufferedImage buffered-image ^String format]
  (let [baos (ByteArrayOutputStream.)]
    (assert (ImageIO/write buffered-image format baos)
            (str "Can't write image in format: " format))
    (.toByteArray baos)))

(defn- content-md5
  "Return the MD5 sum of `bytes`."
  [bytes]
  (base64-encode (digest/md5-bytes bytes)))

(defn- content-type
  "Return the content type of `response`."
  [response]
  (get-in response [:headers "content-type"]))

(defn content-type-extension
  "Returns the filename extension from `content-type`."
  [content-type]
  (if-not (str/blank? content-type)
    (last (str/split content-type #"/"))))

(defn- content-disposition
  "Return the content disposition header of `photo`."
  [photo content-type]
  (str "inline; filename=photo-"
       (:id photo)
       (when-let [title (:title photo)]
         (str "-" (str/replace title #"(?i)[^A-Z0-9]" "-")))
       "."
       (content-type-extension content-type)))

(defn photo-storage-key
  "Return the storage key for `photo` and `content-type`."
  [service photo content-type]
  (let [extension (content-type-extension content-type)]
    (str "photos" "/" (:id photo) "." extension)))

(defn photo-storage-url
  "Return the storage url for `photo`, and `content-type`."
  [service photo content-type]
  (format "https://storage.googleapis.com/%s/%s"
          (-> service :storage :bucket)
          (photo-storage-key service photo content-type)))

(defn- enrich-photo
  "Enrich the `photo` with meta data."
  [service photo]
  (let [{:keys [buffered content-type]} photo]
    (->> {:content-disposition (content-disposition photo content-type)
          :content-length (count (:bytes photo))
          :content-md5 (content-md5 (:bytes photo))
          :content-type content-type
          :height (.getHeight buffered)
          :storage-key (photo-storage-key service photo content-type)
          :url (photo-storage-url service photo content-type)
          :width (.getWidth buffered)}
         (merge photo))))

(defn decode-photo
  "Decode the download `response` of `photo`."
  [service {:keys [id source-url] :as photo} response]
  (let [bytes (IOUtils/toByteArray (:body response))]
    (if-let [buffered (ImageIO/read (ByteArrayInputStream. bytes))]
      (let [content-type (content-type response)]
        (log/infof "Decoded photo %s as content type %s." id content-type)
        (enrich-photo
         service
         (assoc photo
                :buffered buffered
                :bytes bytes
                :content-type content-type)))
      (throw (ex-info (format "Can't decode photo %s: %s" id source-url)
                      {:type :decode-error
                       :photo photo
                       :response response})))))

(defn download-photo
  "Download the original image of `photo`."
  [service {:keys [id source-url] :as photo}]
  (again/with-retries (:retries service)
    (let [{:keys [body status] :as response}
          (http/get
           source-url
           {:as :stream
            :headers {:user-agent (:user-agent service)}})]
      (cond
        (= status 200)
        (decode-photo service photo response)
        :else
        (throw (ex-info (format "Photo download error: Status: %s, Url: %s" status source-url)
                        {:photo photo :response response}))))))

(defn save-photo-to-db
  "Save the `image` to the database."
  [{:keys [db] :as service} photo]
  (merge photo (photos/save! db photo)))

(defn save-photo-to-storage
  "Upload the `image`."
  [service photo]
  (->> {:acl storage/acl-public
        :bytes (:bytes photo)
        :content-type (:content-type photo)
        :cache-control "public, max-age 31536000"
        :content-disposition (:content-disposition photo)
        :content-encoding "identity"
        :key (:storage-key photo)
        :meta-data {:id (str (:id photo))
                    :width (:width photo)
                    :height (:height photo)}}
       (storage/save! (:storage service)))
  (log/infof "Saved photo #%d to Google Storage at %s."
             (:id photo) (:url photo))
  photo)

(defn- save-photo! [db photo]
  (if (:id photo)
    (photos/update! db photo)
    (photos/insert! db photo)))

(defn- update-photo-status
  "Update the `status` of `photo` in the database."
  [service photo status]
  (->> (assoc photo :status (name status))
       (photos/update! (:db service))))

(defrecord Photos [storage user-agent]
  Service
  (-import! [service photo opts]
    (try
      (sql/with-transaction [db (:db service)]
        (let [service (assoc service :db db)
              photo (->> (save-photo! db photo)
                         (download-photo service)
                         (save-photo-to-db service)
                         (save-photo-to-storage service))]
          (log/infof "Photo %s successfully downloaded." (:id photo))
          (update-photo-status service photo :finished)))
      (catch Throwable e
        (let [msg (format "Photo import error: %s" (.getMessage e))]
          (log/error msg)
          (update-photo-status service photo :import-error)
          (throw (ex-info msg {:type ::download-error
                               :photo photo
                               :opts opts})))))))

(defn service
  "Returns a photo service component."
  [& [opts]]
  (-> {:user-agent user-agent
       :retries [100 1000 5000]}
      (merge opts)
      (map->Photos)
      (component/using [:db :storage])))

(defn create-profile-photo!
  "Create the profile `photo` of `user`."
  [{:keys [db] :as service} user photo & [opts]]
  (sql/with-transaction [db db]
    (let [service (assoc service :db db)
          photo (import! service photo opts)]
      (users/add-profile-photos! db user [photo])
      (log/infof "Profile photo %s for user %s successfully created."
                 (:id photo) (:id user))
      photo)))
