(ns burningswell.worker.spots.addresses
  (:require [burningswell.db.spots :as spots]
            [burningswell.worker.api.events :as events]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.geocoder :as geocoder]
            [burningswell.worker.serdes :as serdes]
            [burningswell.worker.topics :as topics :refer [deftopic]]
            [burningswell.worker.util :as util]
            [clojure.pprint :refer [pprint]]
            [com.stuartsierra.component :as component]
            [geocoder.google :as google]
            [jackdaw.streams :as j]
            [sqlingvo.core :as sql]
            [taoensso.timbre :as log]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions])
  (:import org.apache.kafka.streams.kstream.JoinWindows))

(deftopic addresses-updated-topic
  "burningswell.worker.spots.addresses.addresses-updated")

(deftopic spots-table-topic
  "burningswell.worker.spots.addresses.spots-table")

(defn config
  "Returns the configuration for the app."
  [& [opts]]
  (->> {:application
        {"application.id" "burningswell.worker.spots.addresses"
         "bootstrap.servers" (:bootstrap.servers opts)
         "auto.offset.reset" "earliest"
         "cache.max.bytes.buffering" "0"}
        :input {:events topics/api-events-edn
                :locations geocoder/locations-event-topic}
        :output {:locations geocoder/locations-command-topic
                 :spots-table spots-table-topic}}
       (merge opts)))

(defn- spots-created
  "Returns a stream of spot-created event."
  [app builder]
  (-> (j/kstream builder (-> app :config :input :events))
      (events/by-name :burningswell.api.events/spot-created)
      (j/map (fn [[_ {:keys [spot-id] :as event}]]
               (log/infof "Spot created event received: %s" (pr-str event))
               (clojure.pprint/pprint event)
               (let [spot (spots/by-id (:db app) spot-id)]
                 (log/infof "Spot from database: " (pr-str spot))
                 [(util/point->location (:location spot)) spot])))))

(defn- geocode-results
  "Returns the geocode results stream."
  [app builder]
  (j/kstream builder (-> app :config :input :locations)))

(defn- geocode-requests
  "Returns the geocode requests stream."
  [app spots]
  (-> (j/filter spots (fn [[_ {:keys [location]}]] location))
      (j/map (fn [[_ {:keys [location]}]]
               (let [location (util/point->location location)]
                 [location {:location location}])))))

(defn- result->address [result]
  (when-let [location (util/location->point (google/location result))]
    {:city (google/city result)
     :country-id (-> result :country :id)
     :formatted (:formatted-address result)
     :location location
     :postal-code (google/postal-code result)
     :region-id (-> result :region :id)
     :street-name (google/street-name result)
     :street-number (google/street-number result)}))

(defn- save-address!
  "Save the address to the db."
  [db result]
  (when-let [location (util/location->point (google/location result))]
    (let [address (result->address result)]
      (first @(sql/insert db :addresses []
                (sql/values [address])
                (sql/on-conflict [:location]
                  (sql/do-update
                   (sql/excluded-kw-map (seq (disj (set (keys address)) :location)))))
                (sql/returning :*))))))

(defn- save-spot-address!
  "Save the spot address to the db."
  [db spot address]
  (first @(sql/update db :spots
            {:id (:id spot)
             :address-id (:id address)
             :country-id (:country-id address)
             :region-id (:region-id address)}
            (sql/where `(= :id ~(:id spot)))
            (sql/returning :*))))

(defn- save-results!
  [{:keys [db]} spot {:keys [status results]}]
  (log/infof "JOIN: spot = %s, status = %s." (pr-str spot) (pr-str status))
  (when (and (= :ok status) (seq results))
    (let [result (first results)]
      (when-let [address (save-address! db result)]
        (save-spot-address! db spot address)
        (log/infof "Geocoded spot address: %s, %s "
                   (:name spot)
                   (or (:formatted address)
                       (:location address)))))))

(defn- update-country! [db spot]
  (if-let [country (first (countries/within-distance db (:location spot) 5))]
    (spots/update-country! db spot country)
    spot))

(defn- update-region! [db spot]
  (if-let [region (first (regions/within-distance db (:location spot) 5))]
    (spots/update-region! db spot region)
    spot))

(defn build-topology
  "Make the spot photo worker topology."
  [app builder]
  (let [spots-created (spots-created app builder)
        geocode-results (geocode-results app builder)]
    (j/for-each! spots-created
                 (fn [[_ spot]]
                   (log/infof "Spot created: %s" (pr-str spot))
                   (when-not (:country-id spot)
                     (update-country! (:db app) spot))
                   (when-not (:region-id spot)
                     (update-region! (:db app) spot))))
    (-> geocode-results
        (j/peek (fn [[k v]]
                  (log/infof "Geocode results: k= %s, v = %s"
                             (pr-str k)
                             (-> v :results first :formatted-address)))))

    ;; Send geocode requests
    (-> (geocode-requests app spots-created)
        (j/peek (fn [[k v]]
                  (log/infof "Sending geocode request: k = %s, v = %s." (pr-str k) (pr-str v))))
        (j/to (-> app :config :output :locations)))
    ;; Join spots with geocode results
    (j/join-windowed spots-created geocode-results (partial save-results! app)
                     (JoinWindows/of (* 10 1000))
                     {:key-serde (serdes/edn)
                      :value-serde (serdes/edn)}
                     {:key-serde (serdes/edn)
                      :value-serde (serdes/edn)})))

(defrecord Worker [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))
  (stop [app]
    (driver/stop driver app))
  driver/Application
  (config [app]
    (:application config))
  (topology [app builder]
    (build-topology app builder)))

(defn worker
  "Returns a new spot photo worker."
  [& [opts]]
  (-> {:config (config opts)}
      (merge opts)
      (map->Worker)
      (component/using [:db :driver])))
