(ns {{ns-name}}.db.core
  (:require
   [cheshire.core :refer [generate-string parse-string]]
   [clojure.java.jdbc :as jdbc]
   [hikari-cp.core :as hik]
   [{{ns-name}}.config :refer [env]]
   [mount.core :refer [defstate]]
   [honeysql.core :as sql]
   [honeysql.helpers :as h])
  (:import org.postgresql.util.PGobject
           java.sql.Array
           clojure.lang.IPersistentMap
           clojure.lang.IPersistentVector
           [java.sql
            BatchUpdateException
            Date
            Timestamp
            PreparedStatement]))

;;;;;;;;;;;;;;;
;; enable DB ;;
;;;;;;;;;;;;;;;
;; Enablethis to get your DB working
;; (defstate ^:dynamic *db*
;;   :start (hik/make-datasource (-> env :db))
;;   :stop (hik/close-datasource *db*))

(def *db* nil)
;;;;;;;;;;;;;;;



(defn to-date [^java.sql.Date sql-date]
  (-> sql-date (.getTime) (java.util.Date.)))

(extend-protocol jdbc/IResultSetReadColumn
  Date
  (result-set-read-column [v _ _] (to-date v))

  Timestamp
  (result-set-read-column [v _ _] (to-date v))

  Array
  (result-set-read-column [v _ _] (vec (.getArray v)))

  PGobject
  (result-set-read-column [pgobj _metadata _index]
    (let [type  (.getType pgobj)
          value (.getValue pgobj)]
      (case type
        "json" (parse-string value true)
        "jsonb" (parse-string value true)
        "citext" (str value)
        value))))

(extend-type java.util.Date
  jdbc/ISQLParameter
  (set-parameter [v ^PreparedStatement stmt ^long idx]
    (.setTimestamp stmt idx (Timestamp. (.getTime v)))))

(defn to-pg-type [type val]
  (doto (PGobject.)
    (.setType type)
    (.setValue val)))

(defn to-pg-json [value]
  (to-pg-type "jsonb" (generate-string value)))

(defn to-pg-state [value]
  (to-pg-type "state" value))

(extend-type clojure.lang.IPersistentVector
  jdbc/ISQLParameter
  (set-parameter [v ^java.sql.PreparedStatement stmt ^long idx]
    (let [conn      (.getConnection stmt)
          meta      (.getParameterMetaData stmt)
          type-name (.getParameterTypeName meta idx)]
      (if-let [elem-type (when (= (first type-name) \_) (apply str (rest type-name)))]
        (.setObject stmt idx (.createArrayOf conn elem-type (to-array v)))
        (.setObject stmt idx (to-pg-json v))))))

(extend-protocol jdbc/ISQLValue
  IPersistentMap
  (sql-value [value] (to-pg-json value))
  IPersistentVector
  (sql-value [value] (to-pg-json value)))

;; CRUD
(defn dbc! [table entrymap] (jdbc/insert! {:datasource *db*} table entrymap))
(defn dbr [s] (jdbc/query {:datasource *db*} s))
(defn dbu! [table-key valmap where-vec] (jdbc/update! {:datasource *db*} table-key valmap where-vec))
(defn dbd! [table-key s] (jdbc/delete! {:datasource *db*} table-key s))
(defn dbdo! [s] (jdbc/execute! {:datasource *db*} s))
