(ns ca.bigthunder.libs.mq
  "A basic message queue."
  (:require
   [clojure.core.async :as a :refer [<!]]))

(defonce producer    (atom nil))
(defonce publication (atom nil))

(defonce default {:producer-buf-size 20 :consumer-buf-size 5})

(defonce injected (atom default))

(defn- channel [size type]
  (a/chan (case type
           :dropping (a/dropping-buffer size)
           :sliding  (a/sliding-buffer size)
           :fixed    size)))

(defn publish [topic msg]
  "Send msg to all consumers for topic."
  (a/put! @producer {:topic topic :msg msg}))

(def >> publish)

(defn consumer
  [& {:keys [topic action buf-size buf-type]
      :or {buf-size (@injected :consumer-buf-size)
           buf-type :fixed}}]
  "Create a consumer for topic which executes action for each received msg.
   Exit the go block and close channel upon receiving a ::stop! signal."
  (let [c (channel buf-size buf-type)]
    (a/sub @publication topic c)
    (a/go
      (loop [m (<! c)]
        (if-not (= (m :msg) ::stop!)
          (do
           (action m)
           (recur (<! c)))
          (a/close! c))))))

(defn stop-consumers [topic]
  "Send stop signal to all consumers of topic."
  (>> topic ::stop!))

(defn start [& args]
  "Start the message queue."
  (when args
    (reset! injected (first args)))
  (reset! producer
    (channel (@injected :producer-buf-size) :fixed))
  (reset! publication (a/pub @producer :topic)))

(defn stop []
  "Stop the message queue."
  (a/close! @producer)
  (a/unsub-all @publication)
  (reset! producer nil)
  (reset! publication nil))
