(ns {{name}}.core (:gen-class)
  (:require [cascalog.tap :refer [hfs-tap lfs-tap]]
            [cascalog.api :refer [<- ?- stdout select-fields hfs-textline lfs-textline with-job-conf compile-flow combine]]
            [cascalog.ops :refer [sum]]
            [cascalog.more-taps :refer [lfs-delimited hfs-delimited]]
            [clojure.tools.cli :refer [cli]])
  (:import [cascading.avro AvroScheme]
           [org.apache.avro Schema]
           [org.apache.avro Schema$Parser]
           [org.apache.hadoop.fs FileSystem]
           [org.apache.hadoop.fs Path]
           [org.apache.hadoop.conf Configuration]))

(defn parse-int [string]
  (Integer/parseInt string))

(defn query [tsv-input-tap avro-input-tap trap-tap]
  (let [number-1-source (<- [?number]
          (tsv-input-tap :> _ ?number-string)
          (parse-int :< ?number-string :> ?number))
        number-2-source (<- [?number]
          (avro-input-tap :> _ ?number))]
    (<- [?sum]
        (:trap trap-tap)

        ((combine number-1-source number-2-source) :> ?number)
        (sum :< ?number :> ?sum))))

(defn prepare-query [remote? tsv-location avro-location avro-schema-location trap-location]
    (let [fs (FileSystem/get (java.net.URI. avro-schema-location) (Configuration.))
          schema-parser (org.apache.avro.Schema$Parser.)
          schema-stream (.open fs (Path. avro-schema-location))
          schema (.parse schema-parser schema-stream)]
      (query
        ((if remote? hfs-delimited lfs-delimited) tsv-location)
        (let [avro-tap (hfs-tap (AvroScheme. schema) avro-location)]
          (<- [?letter ?number]
              ((select-fields avro-tap ["letter" "number"]) :> ?letter ?number)))
        ((if remote? hfs-textline lfs-textline) trap-location))))

(defn -main [& cmd-args]
  (let [[options args banner]
        (cli cmd-args
             ["-h" "--help" "Show help" :default false :flag true]
             ["-d" "--diagram" "Draw flow diagram." :default false :flag true]
             ["-l" "--local" "Use local data." :default false :flag true]
             ["-h" "--remote" "Use remote data." :default false :flag true])
        [tsv-location avro-location avro-schema-location trap-location output-location] args
        job-name "JOB NAME"
        remote? (:remote options)]

    (when (:help options)
      (println banner)
      (System/exit 0))

    (when (not (or (:local options) (:remote options) (:diagram options)))
      (println "Either local, remote, or diagram must be specified.")
      (System/exit 1))

    (if (:diagram options)
      (.writeDOT
        (compile-flow
          (lfs-textline output-location)
          (prepare-query remote? tsv-location avro-location avro-schema-location trap-location)
          "flow.dot"))
      (?- job-name
          ((if remote? hfs-textline lfs-textline) output-location)
          (prepare-query remote? tsv-location avro-location avro-schema-location trap-location)))))
