;; Copyright (C) 2009-2011 Brendan Ribera. All rights reserved.
;; Distributed under the MIT License; see the file LICENSE
;; at the root of this distribution.
(ns kdtree)

(defrecord Node [left right value])
(defrecord Result [point ^double dist-squared])

(defn- dist-squared
  "Compute the K-dimensional distance between two points"
  [^doubles a ^doubles b]
  (loop [res (double 0.0)
         ind (long 0)]
    (if (== ind (alength a))
      res
      (let [v (- (aget a ind) (aget b ind))]
        (recur (+ res (* v v)) (inc ind))))))

(def ^:private dist-func (atom dist-squared))

(defn set-dist-func! [func]
  (reset! dist-func func))

(defn get-dist-func []
  @dist-func)

(defn- build-tree-internal [points depth]
  (if (empty? points) nil
      (let [point-count (count points)
            k (count (nth points 0))
            dimension (mod depth k)
            points (vec (sort-by #(nth % dimension) points))
            median (quot point-count 2)
            split-point (loop [n median]
                          (cond
                           (= 0 n) n
                           (= (nth (points n) dimension)
                              (nth (points (dec n)) dimension))
                           (recur (dec n))
                           :else n))
            left-tree (build-tree-internal
                       (subvec points 0 split-point)
                       (inc depth))
            right-tree (build-tree-internal
                        (subvec points (inc split-point))
                        (inc depth))]
        (Node. left-tree
               right-tree
               (double-array (nth points split-point))
               (meta (nth points split-point))
               nil))))

(defn build-tree
  "Construct a Kd-tree from points. Assumes all
points are of the same dimension."
  [points]
  (build-tree-internal points 0))



(defn- insert-internal [tree ^doubles point depth point-meta]
  (let [k (alength point)
        dimension (mod depth k)]
    (if (nil? tree)
      (Node. nil nil point point-meta nil)
      (let [^doubles value (:value tree)]
        (if (= (aget point dimension)
               (aget value dimension))
          (Node. (:left tree) (:right tree) value (meta tree) nil)
          (let [go-to-left? (< (aget point dimension)
                               (aget value dimension))
                left (if go-to-left?
                       (insert-internal (:left tree) point (inc depth) point-meta)
                       (:left tree))
                right (if-not go-to-left?
                        (insert-internal (:right tree) point (inc depth) point-meta)
                        (:right tree))]
            (Node. left right value (meta tree) nil)))))))

(defn insert
  "Adds a point to an existing tree."
  [tree point]
  (insert-internal tree (double-array point) 0 (meta point)))



(defn- find-min-internal [tree dimension depth]
  (when tree
    (let [k (count (:value tree))
          min-node (fn [node1 node2]
                     (let [^doubles value1 (:value node1)
                           ^doubles value2 (:value node2)]
                       (if (or (nil? value2)
                               (< (aget value1 dimension)
                                  (aget value2 dimension)))
                         node1 node2)))]
      (if (= dimension (mod depth k))
        ;; if we're at the dimension of interest, follow the left branch or
        ;; take the value - left is always smaller in the current dimension
        (if (:left tree)
          (recur (:left tree) dimension (inc depth))
          tree)
        ;; otherwise, compare min of self & children
        (-> tree
            (min-node (find-min-internal (:left tree) dimension (inc depth)))
            (min-node (find-min-internal (:right tree) dimension (inc depth))))))))

(defn find-min
  "Locate the point with the smallest value in a given dimension.
Used internally by the delete function. Runs in O(√n) time for a
balanced tree."
  [tree dimension]
  (let [res (find-min-internal tree dimension 0)]
      (with-meta (vec (:value res))
                 (meta res))))



(defn- points=
  "Compares 2 points represented by arrays of doubles and return true if they are equal"
  [^doubles a ^doubles b]
  (loop [i 0]
    (cond (== i (alength a)) true
          (== (aget a i) (aget b i)) (recur (inc i))
          :else false)))

(defn- delete-internal
  [tree ^doubles point depth]
  (when tree
    (let [^doubles value (:value tree)
          k (alength value)
          dimension (mod depth k)]
      (cond

       ;; point is not here
       (not (points= point value))
       (let [go-to-left? (< (aget point dimension) (aget value dimension))
             left (if go-to-left?
                    (delete-internal (:left tree) point (inc depth))
                    (:left tree))
             right (if-not go-to-left?
                     (delete-internal (:right tree) point (inc depth))
                     (:right tree))]
         (Node. left right (:value tree) (meta tree) nil))

       ;; point is here... three cases:

       ;; right is not null.
       (:right tree)
       (let [min (find-min-internal (:right tree) dimension (inc depth))]
         (Node.
          (:left tree)
          (delete-internal (:right tree) (:value min) (inc depth))
          (:value min)
          (meta min)
          nil))

       ;; left if not null
       (:left tree)
       (let [min (find-min-internal (:left tree) dimension (inc depth))]
         (Node.
          nil
          (delete-internal (:left tree) (:value min) (inc depth))
          (:value min)
          (meta min)
          nil))

       ;; both left and right are null
       :default nil))))

(defn delete
  "Delete value at the given point. Runs in O(log n) time for a balanced tree."
  [tree point]
  (delete-internal tree (double-array point) 0))



(defn- insert-sorted!
  "Inserts value to sorted transient vector. Vector will not grow
bigger than n elements."
  [vec value ^long n]
  (if (and (== (count vec) n)
           (> (:dist-squared value) (:dist-squared (nth vec (dec n)))))
    vec
    (loop [ind (long 0)
           value value
           vec vec]
     (cond (= ind n) vec
           (= ind (count vec)) (conj! vec value)
           :default (let [existing (nth vec ind)]
                      (if (< (:dist-squared value)
                             (:dist-squared existing))
                        (recur (inc ind) existing (assoc! vec ind value))
                        (recur (inc ind) value vec)))))))

(defn- nearest-neighbor-internal [tree ^doubles point n dimension best]
  (if ;; Empty tree? The best list is unchanged.
         (nil? tree) best

         ;; Otherwise, recurse!
         (let [dimension (long dimension)
               next-dimension (unchecked-remainder-int (unchecked-inc dimension) (alength point))
               ^doubles v (:value tree)
               dim-dist (double (- (aget point dimension)
                                   (aget v dimension)))
               closest-semiplane ((if (> dim-dist 0.0) :right :left) tree)
               farthest-semiplane ((if (> dim-dist 0.0) :left :right) tree)
               ;; Compute best list for the near-side of the search order
               best-with-cur (insert-sorted! best (Result. v
                                                           ((get-dist-func) v point)
                                                           (meta tree)
                                                           nil)
                                             n)
               best-near (nearest-neighbor-internal closest-semiplane point n next-dimension best-with-cur)
               worst-nearest (->> (dec (count best-near))
                                  (nth best-near)
                                  :dist-squared)]

           ;; If the square distance of our search node to point in the
           ;; current dimension is still better than the *worst* of the near-
           ;; side best list, there may be a better solution on the far
           ;; side. Compute & combine with near-side solutions.
           (if (< (* dim-dist dim-dist) worst-nearest)
             (recur farthest-semiplane point n next-dimension best-near)
             best-near))))

(defn nearest-neighbor
  "Compute n nearest neighbors for a point. If n is
omitted, the result is the nearest neighbor;
otherwise, the result is a list of length n."
  ([tree point] (first (nearest-neighbor tree point 1)))
  ([tree point n]
     (->> (transient [])
          (nearest-neighbor-internal tree (double-array point) n 0)
          (persistent!)
          (map #(update-in % [:point] vec)))))



(defn- inside-interval? [interval ^doubles point]
  (let [n (alength point)]
    (loop [ind 0]
      (if (== ind n) true
        (let [^doubles axis-intv (nth interval ind)
              left (double (aget axis-intv 0))
              right (double (aget axis-intv 1))
              value (double (aget point ind))]
          (if (<= left value right)
            (recur (inc ind))
            false))))))

(defn- interval-search-internal [tree interval ^long depth accum]
  (if (nil? tree)
    accum
    (let [^doubles point (:value tree)
          ;; If current points inside interval - add it to accum.
          accum (if (inside-interval? interval point)
                  (conj! accum (with-meta (vec point) (meta tree)))
                  accum)
          k (unchecked-remainder-int depth (alength point))
          dim-value (double (aget point k))
          ^doubles dim-boundaries (nth interval k)
          left-boundary (double (aget dim-boundaries 0))
          right-boundary (double (aget dim-boundaries 1))

          ;;; Go to right subtree only if current dimension value is not to the right of the interval.
          accum (if (<= dim-value right-boundary)
                  (interval-search-internal (:right tree) interval (unchecked-inc depth) accum)
                  accum)

          ;; Go to left subtree only if current dimension is not to the left of the interval.
          accum (if (> dim-value left-boundary)
                  (interval-search-internal (:left tree) interval (unchecked-inc depth) accum)
                  accum)]
      accum)))

(defn interval-search
  "Find all points inside given interval.
Interval is a sequence of boundaries for each dimension.
Example: interval 0≤x≤10, 5≤y≤20 represented as [[0 10] [5 20]]"
  [tree interval]
  (->> (transient [])
       (interval-search-internal tree (vec (map double-array interval)) 0)
       (persistent!)))

