(ns bootstrap.core
    (:require [guarded.core :refer [guarded]]))

(def color-classes #{"primary" "default" "success" "info" "danger" "warning"})

(defn string-keyword?
  [value]
  (or (string? value) (keyword? value)))

(defn kw-str
  [& args]
  (keyword (apply str args)))

(guarded element
  [(tagname string-keyword?) (css keyword?) (opts map?) & content]
      (apply vector (keyword (str (name tagname) (name css))) opts content)
  [(tagname string-keyword?) (css keyword?) & content]
      (apply element tagname css {} content)
  [(tagname string-keyword?) (opts map?) & content]
      (apply element tagname (keyword "") opts content)
  [(tagname string-keyword?) & content]
      (apply element tagname (keyword "") {} content))
    
(guarded button
  [(opts set?) & args]
  (let [color-class (or (some color-classes opts) "default")
        block-class (and (opts "block") ".btn-block")
        tagname (or (some #{"button" "input"} opts) "button")
        css-str (keyword (str ".btn.btn-" color-class block-class))]
       (apply element tagname css-str args))
  [& args] (apply button #{} args))

(defn glyph
  [glyph-name]
  [(keyword (str "span.glyphicon.glyphicon-" glyph-name))])

(guarded navbar
  ; "opts = {:position (fixed-top)
  ;          :style (inverse)
  ;          :container (container-fluid)
  ;          :menu-id (menu)
  ;          :brand-name
  ;          :brand-href (#)}"
  [(opts-set set?) (opts-map map?) & content]
    (let [position (as->
                     #{"fixed-top" "static-top" "fixed-bottom"} $
                     (some $ opts-set)
                     (and $ (str "navbar-" $)))
          style (as->
                  (some #{"inverse"} opts-set) $
                  (or $ "default")
                  (str "navbar-" $))
          menu-id (or (:menu-id opts-map) "menu")
          brand-href (or (:brand-href opts-map) "#")
          container (as->
                      (some #{"fluid"} opts-set) $
                      (and $ (str "-" $))
                      (kw-str ".container" $))]
         (element :nav (kw-str "." style (and position (str "." position)))
           (element :div container
             [:div.navbar-header
               [:button.navbar-toggle.collapsed {:type "button"
                                                 :data-toggle "collapsed"
                                                 :data-target (str "#" menu-id)}
                 [:span.sr-only "Toggle Navigation"]
                 [:span.icon-bar]
                 [:span.icon-bar]
                 [:span.icon-bar]]
               [:a.navbar-brand
                 {:href brand-href}
                 (:brand-name opts-map)]]
             (apply element :div.collapse.navbar-collapse content)
           [:div.collapse.navbar-collapse {:id menu-id}]))))

(guarded col
  [(col-classes map?) & content]
  (as-> (fn [[scr-size [col-size offset]]]
            (str ".col-" scr-size "-" col-size
                 (and offset (str ".col-" scr-size "-offset-" offset)))) $
      (map $ col-classes)
      (apply str $)
      (apply element "div" (keyword $) content)))

;(defn input
;  ([input-type] (input input-type {}))
;  ([input-type attrs-map & content]
;    (%> {:type input-type}
;        #(if (map? attrs-map)
;            (cons (merge attrs-map %) content)
;            (cons % (cons attrs-map content)))
;        #(element :input.form-control %))))

(guarded panel
  [(opts set?) & content]
  (let [color-class (or (some color-classes opts) "default")
        classes (str ".panel.panel-" color-class)]
       (apply element :div classes content))
  [& content]
  (panel #{} content))

(defn panel-body
  [& content]
  (apply element :div.panel-body content))

(defn panel-heading
  [& content]
  (apply element :div.panel-heading content))

(defn panel-title
  [& content]
  (apply element :h3.panel-title content))

(guarded alert
  [(opts set?) (attr-map map?) & content]
  (let [color-class (as-> (some color-classes opts) $ (and $ (str ".alert-" $)))
        dismiss (opts "dismiss")
        classes (str ".alert" color-class (and dismiss ".alert-dismissible"))
        content (if dismiss
                    (cons [:button.close
                           {:type "button" :data-dismiss "alert"}
                           [:span "&times;"]]
                          content)
                    content)
        content (->
                  (merge attr-map {:role "alert"})
                  (cons content))]
    (apply element "div" classes content))
  [(opts set?) & content]
  (apply alert opts {} content))
  
(defn row
  [& content]
  (element :div.row content))

(defn form
  [& args]
  (apply element :form args))

(defn form-group
  [& args]
  (apply element :div :.form-group args))

(defn label
  [& args]
  (apply element :label args))
