(ns bootstrap.core
    (:require [guarded.core :refer [guarded]]))

(def color-classes #{"primary" "default" "success" "info" "danger" "warning"})

(guarded element
  [args-processor (css-str string?) (opts set?) (attrs map?) & content]
    (let [opts-map (or (some #(if (map? %) %) opts) {})
          [tagname css-str* attrs* content-processor] (args-processor opts opts-map)
          tagname (str tagname css-str css-str*)
          attrs (merge attrs* attrs)
          content ((or content-processor identity) content)]
         (apply vector tagname attrs content))
  [args-processor (css-str string?) (opts set?) & content]
    (apply element args-processor css-str opts {} content)
  [args-processor (css-str string?) (attrs map?) & content]
    (apply element args-processor css-str #{} attrs content)
  [args-processor (opts set?) (attrs map?) & content]
    (apply element args-processor "" opts attrs content)
  [args-processor (opts set?) & content]
    (apply element args-processor "" opts {} content)
  [args-processor (attrs map?) & content]
    (apply element args-processor "" #{} attrs content)
  [args-processor & content]
    (apply element args-processor "" #{} {} content))

(defn ^:private element*
  [args-processor] (partial element args-processor))

(def btn
  (element* (fn [opts-set _]
                (let [color-class (or (some color-classes opts-set) "default")
                      block-class (and (opts-set "block") ".btn-block")
                      tagname (or (opts-set "input") "button")
                      css-str (str ".btn.btn-" color-class block-class)]
                     [tagname css-str]))))

(def btn-group
  (element* (fn [opts-set _]
                (let [justified (and (opts-set "justified") (".btn-group-justified"))
                      css-str (str ".btn-group" justified)]
                     ["div" css-str {"role" "group"}]))))

(def glyph
  (element* (fn [opts-set _]
                (let [glyph-name (first opts-set)
                      css-str (str "span.glyphicon.glyphicon-" glyph-name)]
                     ["span" css-str]))))

(def navbar
  (element*
    (fn [opts-set opts-map]
        (let [position (as->
                         #{"fixed-top" "static-top" "fixed-bottom"} $
                         (some $ opts-set)
                         (and $ (str ".navbar-" $)))
              style (as->
                      (get opts-set "inverse" "default") $
                      (str ".navbar-" $))
              menu-id (get opts-map :menu-id "menu")
              brand-href (get opts-map :brand-href "#")
              container (as->
                          (opts-set "fluid") $
                          (and $ (str "-" $))
                          (str ".container" $))
              css-str (str style position)]
            
             ["nav" css-str {}
              (fn [content]
                  [(str "div" container)
                   [:div.navbar-header
                    [:button.navbar-toggle.collapsed {:type "button"
                                                      :data-toggle "collapsed"
                                                      :data-target (str "#" menu-id)}
                     [:span.sr-only "Toggle Navigation"]
                     [:span.icon-bar]
                     [:span.icon-bar]
                     [:span.icon-bar]]
                    [:a.navbar-brand
                      {:href brand-href}
                      (:brand-name opts-map)]]
                   (apply vector :div.collapse.navbar-collapse {:id menu-id} content)])]))))

(def col
  (element*
    (fn [opts-set opts-map]
        (as-> (fn [[scr-size [col-size offset]]]
                  (str ".col-" scr-size "-" col-size
                       (and offset (str ".col-" scr-size "-offset-" offset)))) $
              (map $ opts-map)
              (apply str $)
              ["div" $]))))

(def panel
  (element*
    (fn [opts-set _]
        (let [color-class (or (some color-classes opts-set) "default")
              css-str (str ".panel.panel-" color-class)]
             ["div" css-str]))))

(def panel-body
  (element*
    (fn [_ _]
        ["div" ".panel-body"])))

(def panel-heading
  (element*
    (fn [_ _]
        ["div" ".panel-heading"])))

(def panel-title
  (element*
    (fn [_ _]
        ["h3" ".panel-title"])))

(def alert
  (element*
    (fn [opts-set opts-map]
      (let [color-class (as-> (some color-classes opts-set) $ (and $ (str ".alert-" $)))
            dismiss (and (opts-set "dismiss") (str ".alert-dismissible"))
            css-str (str ".alert" color-class dismiss)
            attrs {:role "alert"}]
           ["div" css-str attrs
            (fn [content]
                (if dismiss
                    (apply vector
                           [:button.close
                            {:type "button" :data-dismiss "alert"}
                            [:span "&times;"]]
                           content)
                    content))]))))

(def row
  (element*
    (fn [_ _] ["div" ".row"])))

(def form
  (element*
    (fn [_ _] ["form"])))

(def form-group
  (element*
    (fn [_ _] ["div" ".form-group"])))

(def label
  (element*
    (fn [_ _] ["label"])))

(def input
  (element*
    (fn [_ _]
        ["input" ".form-control"])))
