(ns ch.codesmith.blocks.postgres
  (:require [ch.codesmith.blocks :as cb]
            [clojure.spec.alpha :as s]
            [migratus.core :as migratus]
            [next.jdbc.connection :as conn])
  (:import (com.zaxxer.hikari HikariDataSource)
           (javax.sql DataSource)))

(defn migrate-db! [^DataSource ds config]
  (let [config (merge {:db    {:datasource ds}
                       :store :database}
                      config)]
    (migratus/init config)
    (migratus/migrate config)))

(defmulti postgres (fn [_ config] (:type config)))

(defn config->db-spec [{:keys [connection-url password username]}]
  {:jdbcUrl                   connection-url
   :username                  username
   :password                  password
   :initializationFailTimeout 10000
   :autoCommit                false})

(s/def ::username string?)
(s/def ::password string?)
(s/def ::connection-url string?)
(s/def ::migratus map?)
(s/def ::external-config (s/keys :req-un [::password ::connection-url]
                                 :opt-un [::username ::migratus]))
(def external-config-checker (cb/checker ::external-config))

(defmethod postgres :external
  [_ config]
  {:instance (let [config               (external-config-checker config)
                   ^HikariDataSource ds (conn/->pool HikariDataSource (config->db-spec config))]
               (migrate-db! ds (:migratus config))
               ds)
   :halt     (fn [^HikariDataSource ds]
               (.close ds))})
