(ns ch.codesmith.blocks.rabbitmq
  (:require [ch.codesmith.blocks.core :as cb]
            [langohr.channel :as lc]
            [langohr.core :as rmq]
            [taoensso.truss :as truss])
  (:import (com.rabbitmq.client Channel Connection)
           (com.rabbitmq.stream Address AddressResolver Environment EnvironmentBuilder)))

(defmethod cb/valid-block? :rmq/connection
  [_ block]
  (instance? Connection block))

(defmethod cb/start-block! ::connection
  [_ _ {:keys [config creds]} _]
  (let [config (cb/deep-merge (cb/ensure-value creds) config)]
    (rmq/connect config)))

(defmethod cb/stop-block! ::connection
  [_ instance]
  (rmq/close instance))

(derive ::connection :rmq/connection)

(defmethod cb/valid-block? :rmq/channel
  [_ block]
  (instance? Channel block))

(defmethod cb/start-block! ::channel
  [_ system _ dynamic-args]
  (lc/open (cb/block system :rmq/connection dynamic-args)))

(defmethod cb/stop-block! ::channel [_ channel]
  (when (not (lc/closed? channel))
    (lc/close channel)))

(derive ::channel :rmq/channel)

(defmacro with-channel [[channel [system dynamic-args]] & body]
  (truss/have! (symbol? channel))
  `(let [system#       ~system
         dynamic-args# ~dynamic-args
         ~channel (cb/block system# :rmq/channel dynamic-args#)]
     (try
       ~@body
       (finally
         (when (lc/closed? ~channel)
           (cb/ensure-block-stopped! system# :rmq/channel dynamic-args#))))))

(defmethod cb/valid-block? :rmq/environment
  [_ block]
  (instance? Environment block))

(defmethod cb/start-block! ::environment
  [_ _ {:keys [config creds]} _]
  ;; TODO@stan: check same names as the normal connection
  (let [{:keys [host port username password vhost
                ssl-context]} (cb/deep-merge (cb/ensure-value creds) config)
        entry-point (Address. host (int port))
        builder     (-> (Environment/builder)
                      (.host (.host entry-point))
                      (.port (.port entry-point))
                      (.addressResolver (reify AddressResolver
                                          (resolve [_ _]
                                            entry-point)))
                      (.username username)
                      (.password password)

                      (.virtualHost vhost))
        builder     (if ssl-context
                      (-> builder
                        (.tls)
                        (.sslContext ssl-context)
                        (.environmentBuilder))
                      builder)]
    (.build ^EnvironmentBuilder builder)))

