(ns ch.codesmith.blocks.rabbitmq.langohr
  (:require [langohr.core :as rmq])
  (:import (com.rabbitmq.client Address Connection ConnectionFactory DefaultSaslConfig ExceptionHandler)
           (com.rabbitmq.client.impl AMQConnection ForgivingExceptionHandler)
           (com.rabbitmq.client.impl.recovery RetryHandler)
           java.util.concurrent.ThreadFactory
           (javax.net SocketFactory)
           (javax.net.ssl SSLContext)))

; # Patch langohr
;
; We want the content of the pull request https://github.com/michaelklishin/langohr/pull/125/files
; and copy here all the necessary files.
; The langohr library contains java code and needs a release to publish (no possibilities to have a git repo)

(defn- platform-string
  []
  (format "Clojure %s on %s %s"
    (clojure-version)
    (System/getProperty "java.vm.name")
    (System/getProperty "java.version")))

(def ^{:private true}
  client-properties {"product"      "Langohr (codesmith patch)"
                     "information"  "See http://clojurerabbitmq.info/"
                     "platform"     (platform-string)
                     "capabilities" (get (AMQConnection/defaultClientProperties) "capabilities")
                     "copyright"    "Copyright (C) 2011-2024 Michael S. Klishin, Alex Petrov"
                     "version"      "5.5.0"})

(defn- auth-mechanism->sasl-config
  [{:keys [authentication-mechanism]}]
  (case authentication-mechanism
    "PLAIN" DefaultSaslConfig/PLAIN
    "EXTERNAL" DefaultSaslConfig/EXTERNAL
    nil))

(defn- ^ConnectionFactory create-connection-factory
  "Creates connection factory from given attributes"
  [settings]
  (let [{:keys [host port username password vhost
                requested-heartbeat connection-timeout ssl ssl-context verify-hostname socket-factory sasl-config
                requested-channel-max thread-factory exception-handler ssl-context-factory
                credentials-provider metrics-collector requested-frame-max handshake-timeout shutdown-timeout
                socket-configurator shutdown-executor heartbeat-executor topology-recovery-executor
                observation-collector credentials-refresh-service recovery-delay-handler nio-params
                channel-should-check-rpc-response-type? work-pool-timeout error-on-write-listener
                topology-recovery-filter connection-recovery-triggering-condition recovered-queue-name-supplier
                traffic-listener use-nio? use-blocking-io? channel-rpc-timeout max-inbound-message-body-size
                connection-name update-client-properties topology-recovery-retry-handler
                update-connection-factory]
         :or   {requested-heartbeat   ConnectionFactory/DEFAULT_HEARTBEAT
                connection-timeout    ConnectionFactory/DEFAULT_CONNECTION_TIMEOUT
                requested-channel-max ConnectionFactory/DEFAULT_CHANNEL_MAX
                sasl-config           (auth-mechanism->sasl-config settings)}} (rmq/normalize-settings settings)
        cf               (ConnectionFactory.)
        final-properties (cond-> client-properties
                           connection-name (assoc "connection_name" connection-name)
                           update-client-properties update-client-properties)
        tls-expected     (or ssl
                           (not (nil? ssl-context))
                           (= port ConnectionFactory/DEFAULT_AMQP_OVER_SSL_PORT))
        final-port       (or port (if tls-expected
                                    ConnectionFactory/DEFAULT_AMQP_OVER_SSL_PORT
                                    ConnectionFactory/DEFAULT_AMQP_PORT))]
    (doto cf
      (.setClientProperties final-properties)
      (.setVirtualHost vhost)
      (.setHost host)
      (.setPort port)
      (.setRequestedHeartbeat requested-heartbeat)
      (.setConnectionTimeout connection-timeout)
      (.setRequestedChannelMax requested-channel-max))
    (if credentials-provider
      (.setCredentialsProvider cf credentials-provider)
      (doto cf
        (.setUsername username)
        (.setPassword password)))
    (when metrics-collector
      (.setMetricsCollector cf metrics-collector))
    (when requested-frame-max
      (.setRequestedFrameMax cf requested-frame-max))
    (when handshake-timeout
      (.setHandshakeTimeout cf handshake-timeout))
    (when shutdown-timeout
      (.setShutdownTimeout cf shutdown-timeout))
    (when socket-configurator
      (.setSocketConfigurator cf socket-configurator))
    (when shutdown-executor
      (.setShutdownExecutor cf shutdown-executor))
    (when heartbeat-executor
      (.setHeartbeatExecutor cf heartbeat-executor))
    (when topology-recovery-executor
      (.setTopologyRecoveryExecutor cf topology-recovery-executor))
    (when topology-recovery-filter
      (.setTopologyRecoveryFilter cf topology-recovery-filter))
    (when observation-collector
      (.setObservationCollector cf observation-collector))
    (when credentials-refresh-service
      (.setCredentialsRefreshService cf credentials-refresh-service))
    (when recovery-delay-handler
      (.setRecoveryDelayHandler cf recovery-delay-handler))
    (when use-nio?
      (.useNio cf))
    (when nio-params
      (.setNioParams cf nio-params))
    (when use-blocking-io?
      (.useBlockingIo cf))
    (when channel-rpc-timeout
      (.setChannelRpcTimeout cf channel-rpc-timeout))
    (when max-inbound-message-body-size
      (.setMaxInboundMessageBodySize cf max-inbound-message-body-size))
    (when (some? channel-should-check-rpc-response-type?)
      (.setChannelShouldCheckRpcResponseType cf channel-should-check-rpc-response-type?))
    (when work-pool-timeout
      (.setWorkPoolTimeout cf work-pool-timeout))
    (when error-on-write-listener
      (.setErrorOnWriteListener cf error-on-write-listener))
    (when connection-recovery-triggering-condition
      (.setConnectionRecoveryTriggeringCondition cf connection-recovery-triggering-condition))
    (when recovered-queue-name-supplier
      (.setRecoveredQueueNameSupplier cf recovered-queue-name-supplier))
    (when traffic-listener
      (.setTrafficListener cf traffic-listener))
    (when sasl-config
      (.setSaslConfig cf sasl-config))
    (when tls-expected
      (if ssl-context
        (do
          (.useSslProtocol cf ^SSLContext ssl-context)
          (.setPort cf final-port))
        (.useSslProtocol cf)))
    (when verify-hostname
      (.enableHostnameVerification cf))
    (when thread-factory
      (.setThreadFactory cf ^ThreadFactory thread-factory))
    (when topology-recovery-retry-handler
      (.setTopologyRecoveryRetryHandler cf ^RetryHandler topology-recovery-retry-handler))
    (when ssl-context-factory
      (.setSslContextFactory cf ssl-context-factory))
    (when socket-factory
      (.setSocketFactory cf ^SocketFactory socket-factory))
    (if exception-handler
      (.setExceptionHandler cf ^ExceptionHandler exception-handler)
      (.setExceptionHandler cf (ForgivingExceptionHandler.)))
    (when update-connection-factory
      (update-connection-factory cf))
    cf))

(defn- address-array-from
  [addresses port]
  (into-array Address
    (map (fn [arg]
           (let [[host port] (if (coll? arg)
                               [(first arg) (second arg)]
                               [arg port])]
             (Address. host port)))
      (remove nil? addresses))))

(defn ^Connection connect
  "Creates and returns a new connection to RabbitMQ."
  ;; defaults
  ([]
   (let [^ConnectionFactory cf (create-connection-factory {})]
     (doto (com.novemberain.langohr.Connection. cf)
       .init)))
  ;; settings
  ([settings]
   (let [settings'             (rmq/normalize-settings settings)
         ^ConnectionFactory cf (create-connection-factory settings')
         xs                    (address-array-from (get settings' :hosts #{})
                                 (get settings' :port))]
     (doto (com.novemberain.langohr.Connection. cf (dissoc settings' :password :username))
       (.init xs)))))
