(ns ch.codesmith.blocks.vault
  (:require [ch.codesmith.blocks.core :as cb]
            [ch.codesmith.blocks.system-values :as cbsv]
            [vault.client.http :as vh]
            [vault.core :as vault]
            [vault.lease :as vl]
            [vault.secrets.kvv2 :as kvv2]))

(defmethod cb/valid-block? :vault/client
  [_ block]
  (extends? vault/SecretEngine block))

(defn ensure-authenticated! [client authenticate-proc]
  (let [auth @(:auth client)]
    (when (vl/expired? auth)
      (authenticate-proc client))
    (when (vl/expires-within? auth 1800)                    ;; 30 min.
      (vault/renew-token client))))

(defn auto-reconnect-vault-client [url creds-type creds]
  (let [authenticate-proc     #(vault/authenticate! % creds-type (cb/ensure-value creds))
        client                (vh/http-client url)
        ensure-authenticated! #(ensure-authenticated! client authenticate-proc)]
    (authenticate-proc client)
    (reify
      vault/SecretEngine
      (list-secrets [_ path]
        (ensure-authenticated!)
        (vault/list-secrets client path))
      (read-secret [_ path opts]
        (ensure-authenticated!)
        (vault/read-secret client path opts))
      (write-secret! [_ path data]
        (ensure-authenticated!)
        (vault/write-secret! client path data))
      (delete-secret! [_ path]
        (ensure-authenticated!)
        (vault/delete-secret! _ path)))))

(defmethod cb/start-block! ::auto-reconnect-client
  [_ _ {:keys [url creds-type creds]
        :or   {creds-type :userpass}} _]
  (auto-reconnect-vault-client url creds-type creds))

(defn read-secret
  ([client secret-path]
   (read-secret
     client
     "secret"
     secret-path
     nil))
  ([client vault-mount secret-path]
   (read-secret client vault-mount secret-path nil))
  ([client vault-mount secret-path opts]
   (kvv2/read-secret client vault-mount secret-path opts)))

(defmethod cbsv/resolve-system-value ::secret
  [system [_ path] dynamic-args]
  (read-secret
    (cb/block system :vault/client dynamic-args)
    (if (ifn? path)
      (path dynamic-args)
      path)))
