(ns ch.codesmith.blocks.core
  (:refer-clojure :exclude [ref])
  (:require [ch.codesmith.blocks.protocols :as p]
            [ch.codesmith.blocks.system-values :as cbsv]
            [clojure.spec.alpha :as s]
            [clojure.walk :as walk]
            [taoensso.telemere :as t]
            [taoensso.truss :refer [have]])
  (:import (clojure.lang IDeref)
           (java.util.concurrent.locks ReentrantLock)
           (java.util.function Supplier)))

(s/def :block/type keyword?)

(s/def :block/config
  (s/keys :req [:block/type]))

;; <editor-fold desc="# Introduction and concepts">
;;
;; We have the following names:
;; - system: an active collection of instantiated components
;; - snapshot: an immutable view of the structure of a system: the current
;; configuration together with the set of active components; note that the
;; components may be stateful and only the structure is immutable.
;; - block: a component in the system. active part of a system: e.g. a connection pool, an http server, ...
;; - blueprints: a family of components that is guaranteed to have common behaviour; for instance
;; the components of the blueprints :jdbc/datasource are guaranteed to be jdbc DataSource.
;; - dynamic args: arguments at the runtime, not config time. -> infinite set of possible
;; components from one configuration.
;;
;; </editor-fold>
;;

(defmacro check [spec value]
  `(let [value# ~value]
     (if-let [data# (s/explain-data ~spec value#)]
       (throw (ex-info (str "Invalid value for spec "
                         (::s/spec data#))
                data#))
       value#)))

;; references and sysvals

(defrecord BlockRef [factory additional-args])

(defn block-ref
  ([factory]
   (block-ref factory {}))
  ([factory additional-args]
   (->BlockRef factory additional-args)))

(defrecord SysValRef [value-spec additional-args])

(defn sysval-ref
  ([value-spec]
   (sysval-ref value-spec {}))
  ([value-spec additional-args]
   (->SysValRef value-spec additional-args)))

(defrecord SysValSupplierRef [value-spec additional-args])

(defn sysval-supplier-ref
  ([value-spec]
   (sysval-supplier-ref value-spec {}))
  ([value-spec additional-args]
   (->SysValSupplierRef value-spec additional-args)))

(defn ensure-value [val-or-supplier]
  (if (instance? Supplier val-or-supplier)
    (.get ^Supplier val-or-supplier)
    val-or-supplier))

;; <editor-fold desc="# Block config functions and multimethods">

(defmulti expand-factory
  {:arglists '([factory config])}
  (fn [factory _]
    factory))

(defmethod expand-factory :default
  ([factory config]
   [[factory config]]))

(defn expand-config [config]
  (into {}
    (mapcat (fn [[factory config]]
              (expand-factory factory config)))
    config))

(defmulti start-block!
  {:arglists '([block-type system config dynamic-args])}
  (fn [block-type _ _ _]
    block-type))

(defmulti resolve-block
  {:arglists '([block-type block])}
  (fn [block-type _]
    block-type))

(defmethod resolve-block :default
  [_ block]
  block)

(defmulti stop-block!
  {:arglists '([block-type block])}
  (fn [block-type _]
    block-type))

(defmethod stop-block! :default
  [_ _])

;; </editor-fold>

(defmethod start-block! :identity [_ _ value _]
  value)

;; <editor-fold desc="# Factory multimethods and system config">

(defmulti valid-block?
  {:arglists '([factory block])}
  (fn [factory _]
    factory))

(defn factory-config [snapshot factory]
  (have (map? snapshot))                                    ;; assert snapshot with clojure deps.
  (or (-> snapshot :config factory :factory/config)
      {:dynamic-args-keys []}))

;; Copied verbatim from the defunct clojure-contrib (http://bit.ly/deep-merge-with)
(defn deep-merge-with [f & maps]
  (apply
    (fn m [& maps]
      (if (every? (some-fn map? nil?) maps)
        (apply merge-with m maps)
        (apply f maps)))
    maps))

(defn deep-merge [& maps]
  (apply deep-merge-with (fn [_ val] val) maps))

;; </editor-fold>

;; <editor-fold desc="# Locking Volatile">
;;
;; Since we want a dynamic/lazy system, we need such system to be a mutable state
;; Atoms and agents are not ideal for system constructions. The formers retry until
;; successful; however starting a component is not stateless and should happen only
;; once. The latters are asynchronous, but we want a synchronous behavior in our
;; case.
;;
;; Accordingly, we implement a constructs with a lock when modifying the state via
;; a volatile variable and a reintrant lock.
;;
;; The construct is called a _locking volatile_.
;;

(deftype LockingVolatile [^:volatile-mutable value ^ReentrantLock reintrant-lock]
  IDeref
  (deref [_] value)
  p/LockingVolatile
  (lock [_]
    (.lock reintrant-lock))
  (unlock [_]
    (.unlock reintrant-lock))
  (unsafe-set! [_ new-value]
    (set! value new-value)))

(defn locking-volatile! [value]
  (->LockingVolatile value (ReentrantLock.)))

(defmacro lvswap! [lvol f & args]
  `(let [lvol# ~lvol]
     (p/lock lvol#)
     (try
       (p/unsafe-set! lvol# (~f (deref lvol#) ~@args))
       (finally
         (p/unlock lvol#)))))

;; </editor-fold>

;; Base implementation

(defn block-key [snapshot factory dynamic-args]
  [factory (select-keys dynamic-args
             (:dynamic-args-keys (factory-config snapshot factory)))])

(defn get-block [snapshot factory dynamic-args]
  (-> snapshot :blocks (get (block-key snapshot factory dynamic-args))))

(declare ensure-block-started!)
(declare ensure-block-stopped!)

(deftype BaseSystem [state]
  p/Systema
  (snapshot [_] @state)
  (ensure-block! [this factory dynamic-args] (ensure-block-started! this factory dynamic-args))
  (stop-block! [this factory dynamic-args] (ensure-block-stopped! this factory dynamic-args))
  (stop-system! [this]
    (p/lock state)
    (try
      (doseq [[factory dynamic-args] (keys (:blocks @state))]
        (p/stop-block! this factory dynamic-args))
      (finally (p/unlock state)))))

(def ^:dynamic *construction-stack* [])

(defn- sysval [system sysval-ref dynamic-args]
  (let [{:keys [value-spec additional-args]} sysval-ref
        dynamic-args (merge dynamic-args additional-args)]
    (if (instance? SysValSupplierRef sysval-ref)
      (reify
        Supplier
        (get [_]
          (cbsv/resolve-system-value system value-spec dynamic-args)))
      (cbsv/resolve-system-value system value-spec dynamic-args))))

(defn resolve-config [system config dynamic-args]
  (walk/postwalk
    (fn [value]
      (condp instance? value
        BlockRef (let [{:keys [factory additional-args]} value]
                   (p/ensure-block! system factory (merge dynamic-args additional-args)))
        SysValRef (sysval system value dynamic-args)
        SysValSupplierRef (sysval system value dynamic-args)
        value))
    config))

(defn block-config [system factory dynamic-args]
  (let [snapshot (p/snapshot system)
        {:keys [default dynamic-args-keys]} (factory-config snapshot factory)
        config   (-> snapshot :config factory)
        config   (deep-merge default
                   (if (empty? dynamic-args-keys)
                     config
                     (let [dynamic-args-keys (have dynamic-args :in dynamic-args-keys)
                           dynamic-args      (select-keys dynamic-args dynamic-args-keys)]
                       (reduce (fn [acc [key value]]
                                 (deep-merge acc (-> config key (get value {}))))
                         {}
                         dynamic-args))))]
    (resolve-config system config dynamic-args)))

(defn- ensure-block-started!′ [^BaseSystem system factory dynamic-args]
  (let [state    (.-state system)
        snapshot @state]
    (when-not (get-block snapshot factory dynamic-args)
      (let [block-key (block-key snapshot factory dynamic-args)]
        (when (some #(= block-key %) *construction-stack*)
          (throw (ex-info "Circular dependency" {:construction-stack *construction-stack*})))
        (let [previous-block-key (first *construction-stack*)]
          (binding [*construction-stack* (conj *construction-stack* [factory dynamic-args])]
            (let [config     (block-config system factory dynamic-args)
                  config     (check :block/config config)
                  block-type (:block/type config)
                  block      (start-block! block-type system config dynamic-args)]
              (lvswap! state
                assoc-in [:blocks block-key]
                {:type       block-type
                 :block      block
                 :dependents (into []
                               (keep identity)
                               [previous-block-key])}))))))))

(defn ensure-block-started! [^BaseSystem system factory dynamic-args]
  (let [state
        (.-state system)

        {:keys [block type]}
        (let [block-key (block-key @state factory dynamic-args)
              get-block (fn []
                          (-> @state :blocks (get block-key)))]
          (if-let [block (get-block)]
            (if-let [dependent (first *construction-stack*)]
              (try                                          ;; we should be in lock already...
                (p/lock state)
                (lvswap! state update-in
                  [:blocks block-key :dependents]
                  conj dependent)
                (get-block)
                (finally
                  (p/unlock state)))
              block)
            (do
              (p/lock state)
              (try
                (ensure-block-started!′ system factory dynamic-args)
                (get-block)
                (finally
                  (p/unlock state))))))]
    (resolve-block type block)))

(defn- ensure-block-stopped!′ [system factory dynamic-args]
  (let [state    (.-state system)
        snapshot @state]
    (when-let [{:keys [type dependents block]} (get-block snapshot factory dynamic-args)]
      (lvswap! state
        (fn [instance]
          (update instance :blocks #(dissoc %
                                      (block-key instance factory dynamic-args)))))
      (doseq [[kind dynamic-args] dependents]
        (p/stop-block! system kind dynamic-args))
      (stop-block! type block))))

(defn ensure-block-stopped! [system factory dynamic-args]
  (t/log! {:data {:factory      factory
                  :dynamic-args dynamic-args}}
    (str "stopping " [factory dynamic-args]))
  (let [state (.-state system)]
    (p/lock state)
    (try
      (ensure-block-stopped!′ system factory dynamic-args)
      (finally
        (p/unlock state))))
  (t/log! {:data {:factory      factory
                  :dynamic-args dynamic-args}}
    (str "stopped" [factory dynamic-args])))

;;; System creation and usage

(defn new-system [config]
  (let [expended-config (expand-config config)]
    (->BaseSystem
      (locking-volatile!
        {:config expended-config
         :blocks {}}))))

(defn stop-system! [system]
  (p/stop-system! system))

(defn project-system [system common-dynamic-args]
  (let [update-dynamic-args #(merge common-dynamic-args %)]
    (reify p/Systema
      (snapshot [_] (p/snapshot system))
      (ensure-block! [_ type dynamic-args]
        (p/ensure-block! system type (update-dynamic-args dynamic-args)))
      (stop-block! [_ type dynamic-args]
        (p/stop-block! system type (update-dynamic-args dynamic-args)))
      (stop-system! [_]
        (p/stop-system! system)))))

(defn block
  ([system factory]
   (block system factory {}))
  ([system factory dynamic-args]
   (p/ensure-block! system factory dynamic-args)))

;; info about a system

(defn started-blocks [system]
  (-> (p/snapshot system)
    :blocks keys))

;; with system-instance

(defmacro with-system
  [[var config] & body]
  `(let [~var (new-system ~config)]
     (try
       ~@body
       (finally
         (stop-system! ~var)))))