(ns ch.codesmith.blocks.system-values
  (:require [clojure.java.io :as io]
            [clojure.java.process :as ps]
            [clojure.string :as str]
            [clojure.tools.reader.edn :as edn]
            [taoensso.telemere :as t]))

;; Dynamic/lazy extraction of values from system.

(defmulti resolve-system-value
  {:arglists '([system [value-type] dynamic-args])}
  (fn [_ [value-type] _]
    value-type))

(defmethod resolve-system-value
  :constant
  [_ [_ value] _]
  value)

(defmethod resolve-system-value
  :env
  [_ [_ value] _]
  (System/getenv value))

(defn op-read [path]
  (t/event! :op-read {:data {:path path}})
  (->> path
    (ps/exec "op" "read")
    str/trim))

(defn op-inject [^String template]
  (t/event! :op-inject {:data {:template template}})
  (let [process (ps/start "op" "inject")
        in      (ps/stdin process)]
    (with-open [wrt (io/writer in)]
      (.write wrt template))
    (let [captured (ps/io-task #(slurp (ps/stdout process)))
          exit     (deref (ps/exit-ref process))]
      (if (zero? exit)
        @captured
        (throw (RuntimeException. (str "Process failed with exit=" exit)))))))

(defn op-path-placeholder [op-path]
  (str "{{ " op-path " }}"))

(defn op-inject-edn [edn-template]
  (edn/read-string (op-inject (pr-str edn-template))))

(defmethod resolve-system-value
  :1password
  [_ [_ path] dynamic-args]
  (op-read (if (ifn? path)
             (path dynamic-args)
             path)))

(defn read-op-credendials [path-prefix]
  (let [op-path (fn [suffix]
                  (str path-prefix "/" suffix))]
    (op-inject-edn
      {:username (op-path-placeholder (op-path "username"))
       :password (op-path-placeholder (op-path "password"))})))

(defmethod resolve-system-value
  :1password-credentials
  [_ [_ path-prefix] dynamic-args]
  (read-op-credendials (if (ifn? path-prefix)
                         (path-prefix dynamic-args)
                         path-prefix)))
