(ns ch.codesmith.blocks
  (:require [integrant.core :as ig]
            [clojure.spec.alpha :as s]))

(s/def ::instance any?)
(s/def ::halt ifn?)
(s/def ::suspend ifn?)
(s/def ::resume ifn?)
(s/def ::resolve-instance ifn?)
(s/def ::inited-block (s/keys :req-un [::instance]
                              :opt-un [::halt ::suspend ::resume ::resolve-instance]))

(s/def ::name keyword?)
(s/def ::components (s/map-of keyword? map?))
(s/def ::system (s/keys :req-un [::name]
                        :opt-un [::components]))

(s/def ::environment keyword?)
(s/def ::profile (s/keys :req-un [::environment]
                         :opt-un [::components]))


(defn checker [spec]
  #(if (s/valid? spec %)
     %
     (throw (ex-info (str "The value " % " is not conform to its spec")
                     {:schema      spec
                      :value       %
                      :explanation (s/explain-data spec %)}))))

(def check-block (checker ::inited-block))
(def check-system (checker ::system))
(def check-profile (checker ::profile))

(defmethod ig/init-key ::block
  [key {::keys [block] :as config}]
  (check-block (block key config)))

(defmethod ig/resolve-key ::block
  [_ {:keys [instance resolve-instance]}]
  (if resolve-instance
    (resolve-instance instance)
    instance))

(defn invoke-optional-method [key block method & args]
  (when-let [operation (block method)]
    (apply operation key (:instance block) args)))

(defmethod ig/halt-key! ::block
  [key block]
  (invoke-optional-method key block :halt))

(defmethod ig/suspend-key! ::block
  [key block]
  (invoke-optional-method key block :suspend))

(defmethod ig/resume-key ::block
  [key block old-value old-impl]
  (invoke-optional-method key block :resume old-value old-impl))

;; some utils

(defn block-identity [_ config]
  {:instance config})

(defn envvar [_ {:keys [envvar]}]
  {:instance (System/getenv envvar)})

(defn block-map [block key-mapper]
  (fn [key {:keys [value] :as config}]
    (let [new-config (reduce-kv (fn [config key value-fn]
                                  (assoc config key (value-fn value)))
                                config
                                key-mapper)]
      (block key new-config))))

(defn block-slurp [block file-keys]
  (fn [key config]
    (let [new-config (reduce-kv (fn [config key value]
                                  (if (file-keys key)
                                    (assoc config key (slurp value))
                                    config))
                                {}
                                config)]
      (block key new-config))))

;; System+profile

(defn system->ig [system profile]
  (let [{name :name system-components :components} (check-system system)
        {environment :environment profile-components :components} (check-profile profile)
        base-config {::name name ::environment environment}]
    (reduce (fn [config key]
              (assoc config key (merge
                                  base-config
                                  (get system-components key)
                                  (get profile-components key))))
            {}
            (into #{}
                  (concat
                    (keys system-components)
                    (keys profile-components))))))

(defn derive-all-blocks [ig-config]
  (let [block-keys (into #{}
                         (comp (filter (fn [[_ config]]
                                         (::block config)))
                               (map first))
                         ig-config)]
    (into {}
          (map (fn [[key config]]
                 [(if (block-keys key)
                    [::block key]
                    key)
                  (update-vals config
                               (fn [value]
                                 (cond
                                   (and (ig/ref? value)
                                        (block-keys (ig/ref-key value)))
                                   (ig/ref [::block (ig/ref-key value)])

                                   (and (ig/refset? value)
                                        (block-keys (ig/ref-key value)))
                                   (ig/refset [::block (ig/ref-key value)])

                                   :else
                                   value)))]))
          ig-config)))

(defn get-block [system-instance block-key]
  (get system-instance [::block block-key]))

(defn resolve-block [system-instance block-key]
  (ig/resolve-key [::block block-key]
                  (get-block system-instance block-key)))

;; Init

(defn init [system profile]
  (let [config (system->ig system profile)
        config (derive-all-blocks config)]
    (-> config
        ig/prep
        ig/init)))

;; with system-instance

(defmacro with-system-instance
  [[var [system profile]] & body]
  `(let [system#  ~system
         profile# ~profile
         ~var (init system# profile#)]
     (try
       ~@body
       (finally
         (ig/halt! ~var)))))