(ns ch.codesmith.blocks
  (:require [clojure.spec.alpha :as s]
            [integrant.core :as ig]))

(s/def ::type keyword?)
(s/def ::config any?)
(s/def ::wrap-fn ifn?)

(s/def ::block (s/keys :opt-un [::type ::config ::wrap-fn]))

(s/def ::name keyword?)
(s/def ::blocks (s/map-of keyword? ::block))
(s/def ::base-config (s/keys :req-un [::name]
                             :opt-un [::blocks]))

(s/def ::profile keyword?)
(s/def ::profile-config (s/keys :req-un [::profile]
                                :opt-un [::blocks]))


(defn checker [spec]
  #(if (s/valid? spec %)
     %
     (throw (ex-info (str "The value " % " is not conform to its spec")
                     {:schema      spec
                      :value       %
                      :explanation (s/explain-data spec %)}))))

(def check-base-config (checker ::base-config))
(def check-profile-config (checker ::profile-config))

;; some utils

(defmethod ig/init-key ::identity [_ value]
  value)

(defn envvar [{:keys [envvar coerce]}]
  (let [value (System/getenv envvar)]
    (if coerce
      (coerce value)
      value)))

(defmethod ig/init-key ::envvar [_ value]
  (envvar value))

(defn slurp-file [{:keys [file slurp] :or {slurp clojure.core/slurp}}]
  (slurp file))

(defmethod ig/init-key ::file [_ value]
  (slurp-file value))

;; system

;; Copied verbatim from the defunct clojure-contrib (http://bit.ly/deep-merge-with)
(defn deep-merge-with [f & maps]
  (apply
    (fn m [& maps]
      (if (every? map? maps)
        (apply merge-with m maps)
        (apply f maps)))
    maps))

(defn deep-merge [& maps]
  (apply deep-merge-with (fn [_ val] val) maps))


;; Init

(defn merge-with-profiles [base & profiles]
  (check-base-config base)
  (doseq [profile profiles]
    (check-profile-config profile))
  {:name     (:name base)
   :profiles (mapv :profile profiles)
   :blocks   (apply deep-merge (:blocks base) (map :blocks profiles))})

(defn halt! [{:keys [system methods derivations]}]
  (try
    (when system (ig/halt! system))
    (finally
      (doseq [key methods]
        (remove-method ig/init-key key))
      (doseq [[child parent] derivations]
        (underive child parent)))))

(defn- init′
  ([{:keys [blocks] :as config}]
   (init′ config (keys blocks)))
  ([{:keys [name profiles blocks]} keys]
   (let [derivations (into []
                           (keep (fn [[key {:keys [type]}]]
                                   (when (nil? type)
                                     (throw (ex-info (str "The key " key " has not type")
                                                     {:key key})))
                                   (when (not (isa? key type))
                                     [key type])))
                           blocks)
         methods (into []
                       (keep (fn [[key {:keys [type trans-fn]}]]
                               (when trans-fn
                                 (when (contains? (methods ig/init-key) key)
                                   (throw (ex-info (str "The key " key " has already a implementation for ig/init-key; not possible to transform arguments")
                                                   {:key  key
                                                    :type type})))
                                 [key type trans-fn])))
                       blocks)
         ig-config (into {}
                         (map (fn [[key {:keys [config]}]]
                                [key config]))
                         blocks)]
     (doseq [[child parent] derivations]
       (derive child parent))
     (doseq [[key type trans] methods]
       (defmethod ig/init-key key [given-key value]
         (ig/init-key type (trans given-key value))))
     (let [methods (mapv first methods)]
       (try
         {:system      (-> ig-config ig/prep (ig/init keys))
          :name        name
          :profiles    profiles
          :derivations derivations
          :methods     methods}
         (catch Exception e
           (halt! {:system      (:system (ex-data e))
                   :derivations derivations
                   :methods     methods})
           (throw e)))))))

(defn init
  ([base+profiles]
   (init′ (apply merge-with-profiles base+profiles)))
  ([base+profiles keys]
   (init′ (apply merge-with-profiles base+profiles) keys)))

(defn get-block [{:keys [system]} key]
  (get system key))

(defn resolve-block [{:keys [system]} key]
  (ig/resolve-key key (get system key)))

;; with system-instance

(defmacro with-system
  [[var base+profiles] & body]
  `(let [~var (init ~base+profiles)]
     (try
       ~@body
       (finally
         (halt! ~var)))))

