(ns cider.nrepl.middleware.spec
  (:require [cider.nrepl.middleware.util.error-handling :refer [with-safe-transport]]
            [cider.nrepl.middleware.util.spec :as spec-utils]
            [cider.nrepl.middleware.util.cljs :as cljs]
            [clojure.string :as str]
            [clojure.walk :as walk]
            [clojure.pprint :as pp]
            [clojure.string :as str]))

(defn str-non-colls
  "Given a form, convert all non collection childs to str."
  [form]
  (walk/postwalk #(if (coll? %)
                    %
                    (str %))
                 form))

(defn spec-list
  "Retrieves a list of all specs in the registry, sorted by ns/name.
  If filter-regex is not empty, keep only the specs with that prefix."
  [filter-regex]
  (let [sorted-specs (->> (spec-utils/registry)
                          keys
                          (map str) 
                          sort)]
    (if (not-empty filter-regex)
      (filter (fn [spec-symbol-str]
                (let [checkable-part (if (.startsWith ^String spec-symbol-str ":")
                                       (subs spec-symbol-str 1)
                                       spec-symbol-str)]
                  (re-find (re-pattern filter-regex) checkable-part)))
              sorted-specs)
      sorted-specs)))


(defn get-multi-spec-sub-specs
  "Given a multi-spec form, call its multi method methods to retrieve
  its subspecs."
  [multi-spec-form]
  (let [[_ multi-method-symbol & _] multi-spec-form]
   (->> @(resolve multi-method-symbol)
        methods 
        (map (fn [[spec-k method]]
               [spec-k (spec-utils/form (method nil))])))))

(defn add-multi-specs
  "Walk down a spec form and for every subform that is a multi-spec
  add its sub specs."
  [form]
  (walk/postwalk (fn [sub-form]
                   (if (and (coll? sub-form)
                            (symbol? (first sub-form))
                            (-> sub-form first name (= "multi-spec")))
                     (concat sub-form (get-multi-spec-sub-specs sub-form))
                     sub-form))
                 form))

(defn spec-from-string
  "Given a string like \"clojure.core/let\" or \":user/email\" returns
  the associated spec in the registry, if there is one."
  [s]
  (let [[spec-ns spec-kw] (str/split s #"/")]
    (if (.startsWith ^String spec-ns ":")
      (spec-utils/get-spec (keyword (subs spec-ns 1) spec-kw))
      (spec-utils/get-spec (symbol s)))))

(defn normalize-spec-fn-form
  "Given a form like (fn* [any-symbol] ... any-symbol...) replace fn* with fn
  and any occurrence of any-symbol with %."
  [[_ [sym] & r]]
  (concat '(clojure.core/fn [%])
        (walk/postwalk (fn [form]
                         (if (and (symbol? form) (= form sym))
                           '%
                           form))
                       r)))

(defn normalize-spec-form
  "Applys normalize-spec-fn-form to any fn* sub form."
  [sub-form]
  (walk/postwalk (fn [form]
                   (if (and (seq? form) (= 'fn* (first form)))
                     (normalize-spec-fn-form form)
                     form))
                 sub-form))

(defn spec-form
  "Given a spec symbol as a string, get the spec form and prepare it for
  a response."
  [spec-name]
  (when-let [spec (spec-from-string spec-name)]
    (-> (spec-utils/form spec)
        add-multi-specs
        normalize-spec-form
        str-non-colls)))

(defn spec-example
  "Given a spec symbol as a string, returns a string with a pretty printed
  example generated by the spec."
  [spec-name]
  (with-out-str
   (-> (spec-from-string spec-name)
       spec-utils/generate
       pp/pprint)))

;; Replies

(defn spec-list-reply [msg]
  {:spec-list (spec-list (:filter-regex msg))})

(defn spec-form-reply [msg]
  {:spec-form (spec-form (:spec-name msg))})

(defn spec-example-reply [msg]
  {:spec-example (spec-example (:spec-name msg))})

(defn handle-spec [handler msg]
  (with-safe-transport handler msg
    "spec-list" spec-list-reply
    "spec-form" spec-form-reply
    "spec-example" spec-example-reply))
