(ns cider.nrepl.middleware.complete
  "Code completion middleware.
  Delegates to the compliment library for the heavy lifting.
  Uses clj-suitable for ClojureScript completion."
  (:require
   [cider.nrepl.middleware.util.cljs :as cljs]
   [cider.nrepl.middleware.util.error-handling :refer [with-safe-transport]]
   [cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.core :as complete]
   [cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.utils :as complete-utils]
   [orchard.misc :as misc]
   [cider.nrepl.inlined.deps.suitable.v0v5v1.suitable.compliment.sources.cljs :as suitable-sources]))

(def shadow-cljs-present?
  (try (require 'shadow.cljs.devtools.api) true
       (catch Throwable _ false)))

;; controls if dynamic cljs code completions are active
(def suitable-enabled? true)

(def suitable-complete-for-nrepl
  (when suitable-enabled?
    (require 'cider.nrepl.inlined.deps.suitable.v0v5v1.suitable.complete-for-nrepl)
    @(resolve 'cider.nrepl.inlined.deps.suitable.v0v5v1.suitable.complete-for-nrepl/complete-for-nrepl)))

(def clj-sources
  "A list of Clojure completion sources for compliment."
  [:cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.special-forms/literals
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.class-members/static-members
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.ns-mappings/ns-mappings
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.resources/resources
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.keywords/keywords
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.local-bindings/local-bindings
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.class-members/members
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.namespaces-and-classes/namespaces-and-classes
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.special-forms/special-forms])

(def cljs-sources
  "A list of ClojureScript completion sources for compliment."
  [::suitable-sources/cljs-source
   ;; The local binding analysis done by
   ;; :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.local-bindings/local-bindings doesn't perform any
   ;; evaluation or execution of the context form. Thus, it is independent of
   ;; the actual host platform differences. Given that, we can use that same
   ;; source for ClojureScript completion.
   :cider.nrepl.inlined.deps.compliment.v0v4v4.compliment.sources.local-bindings/local-bindings])

(defn complete
  [{:keys [ns prefix symbol context extra-metadata enhanced-cljs-completion?] :as msg}]
  ;; TODO: Drop legacy symbol param in version 1.0
  (let [prefix (str (or prefix symbol))
        completion-opts {:ns             (misc/as-sym ns)
                         :context        context
                         :extra-metadata (set (map keyword extra-metadata))}]
    (if-let [cljs-env (cljs/grab-cljs-env msg)]
      ;; ClojureScript completion
      (binding [suitable-sources/*compiler-env* cljs-env]
        ;; First we get whatever candidates we can from the ClojureScript compiler source
        (cond-> (complete/completions prefix (merge completion-opts {:sources cljs-sources}))
          ;; and we optionally append to them dynamically obtained candidates
          ;; See https://github.com/clojure-emacs/clj-suitable#how-does-it-work for details
          (and suitable-enabled? enhanced-cljs-completion?)
          (concat (suitable-complete-for-nrepl (assoc msg :symbol prefix)))))
      ;; Clojure completion
      (complete/completions prefix (merge completion-opts {:sources clj-sources})))))

(defn completion-doc
  [{:keys [ns sym symbol] :as msg}]
  ;; TODO: Drop legacy symbol param in version 1.0
  (let [sym (str (or sym symbol))
        ns (misc/as-sym ns)]
    (if-let [cljs-env (cljs/grab-cljs-env msg)]
      (binding [suitable-sources/*compiler-env* cljs-env]
        (complete/documentation sym ns {:sources cljs-sources}))
      (complete/documentation sym ns {:sources clj-sources}))))

(defn complete-reply [msg]
  {:completions (complete msg)})

(defn doc-reply
  [msg]
  {:completion-doc (completion-doc msg)})

(defn flush-caches-reply
  [_msg]
  (complete-utils/flush-caches)
  {})

(defn handle-complete [handler msg]
  (with-safe-transport handler msg
    "complete" complete-reply
    "complete-doc" doc-reply
    "complete-flush-caches" flush-caches-reply))
